package org.gcube.application.enm.service;

import java.util.UUID;

import org.gcube.application.enm.common.xml.logs.ExperimentLogs;
import org.gcube.application.enm.common.xml.logs.LogType;
import org.gcube.application.enm.common.xml.request.ExperimentRequest;
import org.gcube.application.enm.common.xml.results.ExperimentResults;
import org.gcube.application.enm.common.xml.results.ResultType;
import org.gcube.application.enm.common.xml.status.ExperimentStatus;
import org.gcube.application.enm.service.conn.InformationSystemClient;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.contentmanagement.blobstorage.service.IClient;

/**
 * Execution resource providers must extend this class with the implementation
 * details specific to their underlying platforms. They must override the
 * constructor {@link ExecutionResource#ExecutionResource()}, which is used by 
 * {@link PluginLoader} to load and inject new providers into the service.
 * 
 * @author Erik Torres <ertorser@upv.es>
 */
public abstract class ExecutionResource {

	protected GCUBELog logger = new GCUBELog(ExecutionResource.class);
	protected InformationSystemClient isClient = new InformationSystemClient();

	/**
	 * This default constructor is used by {@link PluginLoader} to load new
	 * execution resource providers into the service.
	 */
	public ExecutionResource() {
		// Setup logging
		logger.trace("Constructor...");
	}

	/**
	 * Submits and experiment to the execution resource.
	 * @param uuid The unequally unique identifier of the experiment.
	 * @param params The parameters of the experiment to submit.
	 * @return The local identifier of the job in the resource.
	 */
	public abstract String sumbitExperiment(final UUID uuid, 
			final ExperimentRequest params);

	/**
	 * Retrieves the status of the job from the execution resource.
	 * @param uuid The unequally unique identifier of the experiment.
	 * @param jobId The local identifier of the job to monitor in the resource.
	 * @return The status of the job.
	 */
	public abstract ExperimentStatus getStatus(final UUID uuid, 
			final String jobId);

	/**
	 * Retrieves the results of the job from the execution resource.
	 * @param uuid The unequally unique identifier of the experiment.
	 * @param jobId The local identifier of the job in the resource.
	 * @return The results of the job from the execution resource.
	 */
	public abstract ExperimentResults getResults(final UUID uuid, 
			final String jobId);

	/**
	 * Retrieves the execution logs of the job from the execution resource.
	 * @param uuid The unequally unique identifier of the experiment.
	 * @param jobId The local identifier of the job in the resource.
	 * @return The execution logs of the job from the execution resource.
	 */
	public abstract ExperimentLogs getLogs(final UUID uuid, final String jobId);

	/**
	 * Cancels the job at the execution resource.
	 * @param uuid The unequally unique identifier of the experiment.
	 * @param jobId The local identifier of the job to cancel in the resource.
	 */
	public abstract void cancelExperiment(final UUID uuid, final String jobId);

	/**
	 * Retrieves the results of the job from the execution resource storying as
	 * a file with the storage service.
	 * @param result The result to store as file.
	 * @param storageClient The storage service client.
	 * @param credentials User credentials.
	 * @return The identifier of the file created in the storage service.
	 */
	public abstract String getResultAsFile(final ResultType result, 
			final IClient storageClient, final String credentials);

	/**
	 * Retrieves the logs of the job from the execution resource storying as
	 * a file with the storage service.
	 * @param log The log to store as file.
	 * @param storageClient The storage service client.
	 * @param credentials User credentials.
	 * @return The identifier of the file created in the storage service.
	 */
	public abstract String getLogAsFile(final LogType log, 
			final IClient storageClient, final String credentials);

	/**
	 * Clean up the output of an experiment from the remote execution site.
	 * @param uuid The unequally unique identifier of the experiment.
	 * @param jobId The local identifier of the job in the resource.
	 */
	public abstract void cleanExperimentOuput(final UUID uuid, 
			final String jobId);

}
