package org.gcube.accounting.datamodel;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Map;
import java.util.Set;

import org.gcube.accounting.exception.InvalidValueException;

public interface UsageRecord extends Comparable<UsageRecord> {
	
	public enum OperationResult {
		SUCCESS, FAILED
	}
	
	/**
	 * @return a Set containing the keys of required fields
	 */
	public Set<String> getRequiredFields();
	
	/**
	 * Return the Usage Record Type
	 * @return {#UsageRecord} Type
	 */
	public String getUsageRecordType();
	
	/**
	 * Return the unique id for this {#UsageRecord}
	 * @return {#UsageRecord} Unique ID
	 */
	public String getId();

	/**
	 * The ID is automatically Created by the implementation Class. 
	 * Set the ID only if you really know what you are going to do.
	 * Set the unique id for this {#UsageRecord}
	 * @param id {#UsageRecord} Unique ID
	 * @throws InvalidValueException
	 */
	public void setId(String id) throws InvalidValueException;
	
	/**
	 * Return the user (or the Consumer Identity, that in the S2S 
	 * communication is another service)
	 * @return Consumer ID
	 */
	public String getConsumerId();

	/**
	 * Set the user (or the Consumer Identity, that in the S2S 
	 * communication is another service)
	 * @param consumerId Consumer ID
	 * @throws InvalidValueException
	 */
	public void setConsumerId(String consumerId) throws InvalidValueException;
	
	/**
	 * Return the instant when this {#UsageRecord} was created.
	 * @return the creation time for this {#UsageRecord}
	 */
	public Calendar getCreationTime();

	/**
	 * The CreationTime is automatically created by the implementation Class. 
	 * Set the CreationTime only if you really know what you are going to do.
	 * Set the instant when this {#UsageRecord} was created.
	 * @param creationTime creation time
	 * @throws InvalidValueException
	 */
	public void setCreationTime(Calendar creationTime) throws InvalidValueException;
	
	
	/**
	 * Return the scope of this {#UsageRecord} 
	 * @return The scope of this {#UsageRecord} 
	 */
	public String getScope();
	
	/**
	 * Set the accounting scope of the {#UsageRecord} 
	 * @param scope The accounting scope of the {#UsageRecord}
	 * @throws InvalidValueException
	 */
	public void setScope(String scope) throws InvalidValueException;
	
	/**
	 * Return all resource-specific properties. The returned Map is a copy of
	 * the internal representation. Any modification to the returned Map MUST
	 * not affect the object
	 * @return a Map containing the properties
	 */
	public Map<String, Serializable> getResourceProperties();

	/**
	 * Set all resource-specific properties, replacing existing ones
	 */
	public void setResourceProperties(Map<String, Serializable> resourceSpecificProperties) throws InvalidValueException;

	/**
	 * Return the value of the given resource property.
	 * @param key the key of the requested property 
	 * @return the value of the given resource property
	 */
	public Serializable getResourceProperty(String key);
	
	/**
	 * Set the value of the given resource property.
	 * If the key has the value of one of the predefined property, the value
	 * is validated.
	 * @param key the key of the requested property 
	 * @param value the value of the given resource property
	 */
	public void setResourceProperty(String key, Serializable value) throws InvalidValueException;
	
	/**
	 * @return the Operation Result of the accounted operation.
	 */
	public OperationResult getOperationResult();
	
	/**
	 * Set the Operation Result related to the accounted Usage Record
	 * @param operationResult the Operation Result to set
	 * @throws InvalidValueException 
	 */
	public void setOperationResult(OperationResult operationResult) throws InvalidValueException;
	
	
	/**
	 * Validate the Resource Record.
	 * The validation check if all the Required Field are set and has valid 
	 * value.
	 * @throws InvalidValueException
	 */
	public void validate() throws InvalidValueException;

}