package org.gcube.accounting.datamodel;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Map;

import org.gcube.accounting.exception.InvalidValueException;

public interface UsageRecord extends Comparable<UsageRecord>{
	
	public enum OperationResult {
		SUCCESS, FAILED
	}
	
	/**
	 * Return the unique id for this {#UsageRecord}
	 * @return {#UsageRecord} Unique ID
	 */
	public String getId();

	/**
	 * Set the unique id for this {#UsageRecord}
	 * @param id {#UsageRecord} Unique ID
	 * @throws InvalidValueException
	 */
	public void setId(String id) throws InvalidValueException;
	
	/**
	 * Return the identity of the entity creating this {#UsageRecord}
	 * @return Creator ID
	 */
	public String getCreatorId();

	/**
	 * Set the identity of the entity creating this {#UsageRecord}
	 * @param creatorId Creator ID
	 * @throws InvalidValueException
	 */
	public void setCreatorId(String creatorId) throws InvalidValueException;
	
	
	/**
	 * Return the identity of the entity that consumed the resource
	 * @return Consumer ID
	 */
	public String getConsumerId();

	/**
	 * Set the identity of the entity that consumed the resource
	 * @param consumerId Consumer ID
	 * @throws InvalidValueException
	 */
	public void setConsumerId(String consumerId) throws InvalidValueException;
	
	/**
	 * Return the creation time for this {#UsageRecord}
	 * @return the creation time for this {#UsageRecord}
	 */
	public Calendar getCreationTime();

	/**
	 * Set the creation time for this {#UsageRecord}
	 * @param creationTime creation time
	 * @throws InvalidValueException
	 */
	public void setCreationTime(Calendar creationTime) throws InvalidValueException;
	
	
	/**
	 * Return the accounting scope of the {#UsageRecord} 
	 * @return The Accounting scope of the {#UsageRecord} 
	 */
	public String getResourceScope();
	
	/**
	 * Set the accounting scope of the {#UsageRecord} 
	 * @param scope The accounting scope of the {#UsageRecord}
	 * @throws InvalidValueException
	 */
	public void setResourceScope(String scope) throws InvalidValueException;
	
	/**
	 * Return the id of the usage record aggregating this, null if this record
	 * has not been aggregated by any record.
	 * @return Aggregated Id The ID of the aggregation Record
	 */
	public String getAggregatedUsageRecordId();

	/**
	 * Set the id of the usage record aggregating this
	 * @param aggregatedId The ID of the aggregation Record
	 * @throws InvalidValueException
	 */
	public void setAggregatedUsageRecordId(String aggregatedId)  throws InvalidValueException;
	
	/**
	 * Return all resource-specific properties. The returned Map is a copy of
	 * the internal representation. Any modification to the returned Map MUST
	 * not affect the object
	 * @return a Map containing the properties
	 */
	public Map<String, Serializable> getResourceProperties();

	/**
	 * Set all resource-specific properties, replacing existing ones
	 */
	public void setResourceProperties(Map<String, Serializable> resourceSpecificProperties) throws InvalidValueException;
	

	/**
	 * Return the value of the given resource property.
	 * @param key the key of the requested property 
	 * @return the value of the given resource property
	 */
	public Serializable getResourceProperty(String key);
	
	/**
	 * Set the value of the given resource property.
	 * If the key has the value of one of the predefined property, the value
	 * is validated.
	 * @param key the key of the requested property 
	 * @param value the value of the given resource property
	 */
	public void setResourceProperty(String key, Serializable value) throws InvalidValueException;
	
	/**
	 * @return the Operation Result related to the accounted Usage Record
	 */
	public OperationResult getOperationResult();
	
	/**
	 * Set the Operation Result related to the accounted Usage Record
	 * @param operationResult the Operation Result to set
	 * @throws InvalidValueException 
	 */
	public void setOperationResult(OperationResult operationResult) throws InvalidValueException;
	
	
	/**
	 * Validate the Resource Record
	 * @throws InvalidValueException
	 */
	public void validate() throws InvalidValueException;

}