

package org.gcube.accounting.common.file;

import java.io.File;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;

import org.apache.log4j.Logger;

public class FileWatcher {

	private static Logger logger = Logger.getLogger(FileWatcher.class);

	private File file;
	private FileChangedListener listener;
	private long interval;
	private Timer timer;

	public FileWatcher(String fileName) {
		this(new File(fileName));
	}

	public FileWatcher(File file) {
		this.file = file;
		this.setInterval(10000L);
	}

	public void setListener(FileChangedListener listener) {
		this.listener = listener;
	}

	public File getFile() {
		return this.file;
	}

	/**
	 * Return the interval between checks
	 * 
	 * @return the interval in milliseconds
	 */
	public long getInterval() {
		return interval;
	}

	/**
	 * Set the check interval
	 * 
	 * @param interval
	 *            in milliseconds
	 */
	public void setInterval(long interval) {
		this.interval = interval;
		this.scheduleFileCheck();
	}

	private void scheduleFileCheck() {
		this.stop();
		TimerTask task = new CheckModificationTask(this.file);
		this.timer = new Timer();
		timer.schedule(task, new Date(), this.getInterval());
	}
	
	public void stop() {
		if (this.timer != null) {
			this.timer.cancel();
			this.timer.purge();
		}
	}

	class CheckModificationTask extends TimerTask {

		/**
		 * The file to check
		 */
		private File file;

		/**
		 * When the file was last checked
		 */
		private long timestamp;

		public CheckModificationTask(File file) {
			this.file = file;
			this.timestamp = 0L;
		}

		/**
		 * Check file for modifications. If modified, notify the listener
		 */
		public final void run() {
			logger.debug("checking file '" + file.getName()
					+ "' for modifications");
			long timestamp = file.lastModified();
			if (this.timestamp != timestamp) {
				this.timestamp = timestamp;
				if (listener != null)
					listener.fileChanged(file);
			}
		}

	}

}
