/**
 * (c) 2014 FAO / UN (project: fi-security-server)
 */
package org.fao.fi.security.server.providers.validators.token.impl;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import javax.inject.Singleton;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheException;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import net.sf.ehcache.config.CacheConfiguration;
import net.sf.ehcache.event.CacheEventListener;
import net.sf.ehcache.store.MemoryStoreEvictionPolicy;

import org.fao.fi.security.server.providers.validators.token.impl.configuration.TokenManagerSimpleConfiguration;
import org.fao.fi.security.server.providers.validators.token.spi.AbstractTokenManager;
import org.fao.fi.security.server.providers.validators.token.spi.TokenManagerConfigurator;

/**
 * Place your class / interface description here.
 *
 * History:
 *
 * ------------- --------------- -----------------------
 * Date			 Author			 Comment
 * ------------- --------------- -----------------------
 * 30 Apr 2014   Fiorellato     Creation.
 *
 * @version 1.0
 * @since 30 Apr 2014
 */
@Singleton
public class LocalTokenManager extends AbstractTokenManager implements CacheEventListener {
	static final protected String TOKEN_CACHE_ID = "token.cache";
	static final protected int TOKEN_CACHE_SIZE = 10000;
	static final protected int TOKEN_EXPIRATION_TIME = 5;

	protected TokenManagerConfigurator _configuration;
	
	protected CacheManager _cacheManager;
	protected Cache _tokenCache;
	
	public LocalTokenManager() {
	}
	
	@Inject public LocalTokenManager(TokenManagerConfigurator configuration) {
		this._configuration = configuration;
	}
	
	@PostConstruct
	public void completeInitialization() {
		if(this._configuration == null) {
			this._log.warn("No token manager configuration provided for {}: using defaults...", this);
			
			this._configuration = new TokenManagerSimpleConfiguration(TOKEN_CACHE_ID, TOKEN_CACHE_SIZE, TOKEN_EXPIRATION_TIME);
		}
		
		this._cacheManager = CacheManager.getInstance();
		
		if(this._cacheManager == null)
			this._cacheManager = CacheManager.create();
		
		String cacheId = this._configuration.getTokenCacheId();
		
		this._log.info("Initializing {} with {} and cache ID {}", this, this._cacheManager, cacheId);
		
		if(!this._cacheManager.cacheExists(cacheId)) {
			CacheConfiguration cacheConfiguration = new CacheConfiguration(cacheId, this._configuration.getTokenCacheSize());
			cacheConfiguration.eternal(false).
							   timeToIdleSeconds(this._configuration.getTokenExpirationTime()).
							   timeToLiveSeconds(this._configuration.getTokenExpirationTime()).
							   memoryStoreEvictionPolicy(MemoryStoreEvictionPolicy.FIFO).
							   overflowToOffHeap(false);
	
			this._tokenCache = new Cache(cacheConfiguration);
			this._tokenCache.getCacheEventNotificationService().registerListener(this);
			
			this._cacheManager.addCacheIfAbsent(this._tokenCache);
		}

		this._log.info("{} has been initialized with {}", this.getClass(), this._configuration.getClass());
	}
	
	public void updateCacheTimeout(int timeout) {
		this._tokenCache.getCacheConfiguration().setTimeToIdleSeconds(timeout);
	}
	
	public Cache getCache() {
		return this._tokenCache;
	}
	
	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.providers.validators.token.spi.TokenManager#tokenAlreadyAvailable(java.lang.String)
	 */
	@Override
	public boolean exists(String token) {
		return this._tokenCache.get(token) != null && 
			   this._tokenCache.isKeyInCache(token);
	}
	
	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.providers.validators.token.spi.TokenManager#storeNewToken(java.lang.String)
	 */
	@Override
	public void store(String token) {
		this._tokenCache.put(new Element(token, System.currentTimeMillis()));
	}
	
	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.providers.validators.token.spi.TokenManager#removeToken(java.lang.String)
	 */
	@Override
	public boolean remove(String token) {
		return this._tokenCache.remove(token);
	}
	
	/* (non-Javadoc)
	 * @see net.sf.ehcache.event.CacheEventListener#notifyElementEvicted(net.sf.ehcache.Ehcache, net.sf.ehcache.Element)
	 */
	@Override
	public void notifyElementEvicted(Ehcache cache, Element element) {
		this._log.info("Element {} has been evicted from cache {}", element.getObjectKey(), cache.getName());
	}
	
	/* (non-Javadoc)
	 * @see net.sf.ehcache.event.CacheEventListener#notifyElementExpired(net.sf.ehcache.Ehcache, net.sf.ehcache.Element)
	 */
	@Override
	public void notifyElementExpired(Ehcache cache, Element element) {
		this._log.info("Element {} of cache {} has expired", element.getObjectKey(), cache.getName());
	}
	
	/* (non-Javadoc)
	 * @see net.sf.ehcache.event.CacheEventListener#notifyElementPut(net.sf.ehcache.Ehcache, net.sf.ehcache.Element)
	 */
	@Override
	public void notifyElementPut(Ehcache cache, Element element) throws CacheException {
		this._log.info("Putting element {} in cache {}", element.getObjectKey(), cache.getName());
	}
	
	/* (non-Javadoc)
	 * @see net.sf.ehcache.event.CacheEventListener#notifyElementRemoved(net.sf.ehcache.Ehcache, net.sf.ehcache.Element)
	 */
	@Override
	public void notifyElementRemoved(Ehcache cache, Element element) throws CacheException {
		this._log.warn("Element {} of cache {} has been removed", element.getObjectKey(), cache.getName());
	}
	
	/* (non-Javadoc)
	 * @see net.sf.ehcache.event.CacheEventListener#notifyElementUpdated(net.sf.ehcache.Ehcache, net.sf.ehcache.Element)
	 */
	@Override
	public void notifyElementUpdated(Ehcache cache, Element element) throws CacheException {
		this._log.warn("Element {} of cache {} has been updated", element.getObjectKey(), cache.getName());
	}
	
	/* (non-Javadoc)
	 * @see net.sf.ehcache.event.CacheEventListener#notifyRemoveAll(net.sf.ehcache.Ehcache)
	 */
	@Override
	public void notifyRemoveAll(Ehcache cache) {
		this._log.warn("Removing all elements from cache {}", cache.getName());
	}
	
	/* (non-Javadoc)
	 * @see net.sf.ehcache.event.CacheEventListener#dispose()
	 */
	@Override
	public void dispose() {
		this._log.warn("Disposing caches...");
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#clone()
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#finalize()
	 */
	@Override
	protected void finalize() throws Throwable {
		super.finalize();
		
		this._cacheManager.shutdown();
	}

	protected String getScopedToken(String scope, String token) {
		return scope + "_" + token;
	}
}
