/**
 * (c) 2014 FAO / UN (project: fi-security-server)
 */
package org.fao.fi.security.server.providers.validators.token.spi;

import java.util.List;

import javax.ws.rs.container.ContainerRequestContext;

import org.fao.fi.security.common.services.exceptions.BadRequestException;
import org.fao.fi.security.common.services.exceptions.token.InvalidTokenException;
import org.fao.fi.security.common.services.exceptions.token.MissingTokenException;
import org.fao.fi.security.common.services.exceptions.token.TokenProcessingException;
import org.fao.fi.security.common.services.spi.token.TokenExchangeConstants;
import org.fao.fi.security.common.support.token.spi.TokenProcessor;
import org.fao.fi.security.common.utilities.LoggingClient;
import org.fao.fi.security.server.providers.validators.SecuredResourceValidationServiceProvider;

/**
 * Place your class / interface description here.
 *
 * History:
 *
 * ------------- --------------- -----------------------
 * Date			 Author			 Comment
 * ------------- --------------- -----------------------
 * 30 Apr 2014   Fiorellato     Creation.
 *
 * @version 1.0
 * @since 30 Apr 2014
 */
abstract public class AbstractTokenConsumer extends LoggingClient implements TokenConsumer, SecuredResourceValidationServiceProvider {
	private TokenManager _tokenManager;
	private TokenProcessor _tokenProcessor;
		
	/**
	 * Class constructor
	 *
	 * @param tokenManager
	 * @param tokenProcessor
	 */
	public AbstractTokenConsumer(TokenManager tokenManager, TokenProcessor tokenProcessor) {
		super();
		this._tokenManager = tokenManager;
		this._tokenProcessor = tokenProcessor;
	}

	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.providers.token.spi.TokenConsumer#passthrough(javax.ws.rs.container.ContainerRequestContext)
	 */
	@Override
	public synchronized boolean validate(ContainerRequestContext request) throws BadRequestException {
		try {
			this.consumeTokenIfValid(this.process(this.extractToken(request)));
			
			return true;
		} catch (InvalidTokenException ITe) {
			return false;
		}catch (Throwable t) {
			throw new BadRequestException(t.getMessage());
		}
	}

	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.providers.spi.TokenConsumer#consumeTokenIfValid(java.lang.String)
	 */
	@Override
	public synchronized void consumeTokenIfValid(String token) throws InvalidTokenException {
		if(this.isTokenValid(token)) {
			this._tokenManager.remove(token);
		} else 
			throw new InvalidTokenException("Provided token is invalid!");
	}
	
	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.providers.token.spi.TokenConsumer#extractToken(javax.ws.rs.container.ContainerRequestContext)
	 */
	@Override
	public String extractToken(ContainerRequestContext request) throws MissingTokenException {
		List<String> tokenHeaders = request.getHeaders().get(TokenExchangeConstants.TOKEN_HEADER);
		
		if(tokenHeaders == null || tokenHeaders.isEmpty())
			throw new MissingTokenException("No token header available");
		
		String tokenHeader = null;
		
		for(String header : tokenHeaders) {
			if(header != null) {
				tokenHeader = header;
				break;
			}
		}
		
		if(tokenHeader == null || "".equals(tokenHeader))
			throw new MissingTokenException("Token header is available but it has no value");
		
		return tokenHeader;
	}

	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.providers.spi.TokenConsumer#isTokenValid(java.lang.String)
	 */
	@Override
	public synchronized boolean isTokenValid(String token) throws InvalidTokenException {
		return token != null && this._tokenManager.exists(token);
	}
	
	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.providers.token.spi.TokenConsumer#process(java.lang.String)
	 */
	@Override
	public String process(String token) throws TokenProcessingException {
		long end, start = System.currentTimeMillis();
		try {
			this._log.info("Beginning token processment...");
			return this._tokenProcessor.processBeforeConsumption(token);
		} finally {
			end = System.currentTimeMillis();
			
			this._log.info("Token processment has completed in {} mSec.", end - start);
		}
	}
}