/**
 * (c) 2014 FAO / UN (project: fi-security-common)
 */
package org.fao.fi.security.common.utilities.pgp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.security.SecureRandom;
import java.util.Iterator;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPCompressedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedDataGenerator;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPLiteralData;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPPublicKeyRingCollection;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.jcajce.JcePGPDataEncryptorBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcePublicKeyKeyEncryptionMethodGenerator;
import org.fao.fi.security.common.encryption.pgp.exceptions.KeyringAccessException;
import org.fao.fi.security.common.encryption.pgp.exceptions.KeyringException;
import org.fao.fi.security.common.utilities.FileUtils;

public class PGPEncryptor extends AbstractPGPProcessor {
	/**
	 * Class constructor
	 */
	public PGPEncryptor() {
	}
	
	public byte[] encryptBytes(byte[] toEncrypt, File publicKeyFile) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		return this.encryptBytes(toEncrypt, new FileInputStream(publicKeyFile));
	}
	
	public byte[] encryptBytes(byte[] toEncrypt, InputStream publicKeyStream) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		return this.encryptBytes(toEncrypt, this.readPublicKeyFromKeyringCollection(publicKeyStream));
	}
	
	public byte[] encryptBytes(byte[] toEncrypt, PGPPublicKey publicKey) throws IOException, PGPException, NoSuchProviderException {
		ByteArrayInputStream byteInStream = new ByteArrayInputStream(toEncrypt);
		ByteArrayOutputStream byteOutStream = new ByteArrayOutputStream();
		
		try {
			this.encryptStream(byteInStream, byteOutStream, publicKey);
			
			byteOutStream.flush();
			byteOutStream.close();
			
			return byteOutStream.toByteArray();
		} finally {
			byteInStream.close();
		}
	}

	public void encryptStream(InputStream toEncrypt, OutputStream encrypted, File publicKeyFile) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		this.encryptStream(toEncrypt, encrypted, new FileInputStream(publicKeyFile));
	}

	public void encryptStream(InputStream toEncrypt, OutputStream encrypted, InputStream publicKeyStream) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		this.encryptStream(toEncrypt, encrypted,  this.readPublicKeyFromKeyringCollection(publicKeyStream));
	}
	
	public void encryptStream(InputStream toEncrypt, OutputStream encrypted, PGPPublicKey publicKey) throws IOException, PGPException, NoSuchProviderException {
		this._log.debug(" * Key Strength   = {}", publicKey.getBitStrength());
		this._log.debug(" * Algorithm      = {}", publicKey.getAlgorithm());
		this._log.debug(" * Bit strength   = {}", publicKey.getBitStrength());
		this._log.debug(" * Version        = {}", publicKey.getVersion());
		this._log.debug(" * Encryption key = {}", publicKey.isEncryptionKey());
		this._log.debug(" * Master key     = {}", publicKey.isMasterKey());
		
		int numKeys = 0;
		
		@SuppressWarnings("unchecked")
		Iterator<String> userIDs = (Iterator<String>)publicKey.getUserIDs();
		
		while(userIDs.hasNext()) {
			this._log.debug(" * User ID        = {}",userIDs.next());
			numKeys++;
		}
		
		this._log.debug(" * Key Count      = {}", numKeys);
		
		this.doEncryptStream(toEncrypt, encrypted, publicKey);
	}
	
	public PGPPublicKey readPublicKeyFromKeyringCollection(File publicKeyFile) throws IOException, KeyringAccessException, PGPException {
		return this.readPublicKeyFromKeyringCollection(new FileInputStream(publicKeyFile));
	}
	
	@SuppressWarnings("unchecked")
	public PGPPublicKey readPublicKeyFromKeyringCollection(InputStream publicKeyStream) throws IOException, KeyringAccessException, PGPException {
		try {
			PGPPublicKeyRing publicKeyRing = null;
		
			PGPPublicKeyRingCollection publicKeyRingCollection = new PGPPublicKeyRingCollection(publicKeyStream);
			
			this._log.debug(" * Key ring size = {}", publicKeyRingCollection.size());
			
			Iterator<PGPPublicKeyRing> keyRingIterator = (Iterator<PGPPublicKeyRing>)publicKeyRingCollection.getKeyRings();
			Iterator<PGPPublicKey> publicKeysIterator;
			
			PGPPublicKey publicKey;
			
			while(keyRingIterator.hasNext()) {
				publicKeyRing = (PGPPublicKeyRing) keyRingIterator.next();
				
				publicKeysIterator = (Iterator<PGPPublicKey>)publicKeyRing.getPublicKeys();
				
				while (publicKeysIterator.hasNext()) {
					publicKey = (PGPPublicKey) publicKeysIterator.next();
					
					if (publicKey.isEncryptionKey())
						return publicKey;
				}
			}
			
			throw new KeyringAccessException("Unable to extract public key from keyring");
		} finally {
			if(publicKeyStream != null)
				publicKeyStream.close();
		}
	}

	private long doEncryptStream(InputStream toEncrypt, OutputStream encryptedStream, PGPPublicKey publicKey) throws IOException, PGPException, NoSuchProviderException {
		File tempFile = File.createTempFile("plain", "tpgp");
		OutputStream tempFileStream = new FileOutputStream(tempFile);
		
		try {
			tempFile.deleteOnExit();
			
			FileUtils.pipeStreams(toEncrypt, tempFileStream);
			
			tempFileStream.flush();
			
			encryptedStream = new DataOutputStream(encryptedStream);
			
			ByteArrayOutputStream targetOutputStream = new ByteArrayOutputStream();
			
			PGPCompressedDataGenerator compressedDataGenerator = new PGPCompressedDataGenerator(PGPCompressedDataGenerator.ZIP);
			PGPUtil.writeFileToLiteralData(compressedDataGenerator.open(targetOutputStream), PGPLiteralData.BINARY, tempFile);
			compressedDataGenerator.close();
			
			Provider provider = new BouncyCastleProvider();
			SecureRandom random = new SecureRandom();
			
			PGPEncryptedDataGenerator encryptor = 
				new PGPEncryptedDataGenerator(
					new JcePGPDataEncryptorBuilder(PGPEncryptedDataGenerator.CAST5).
							setSecureRandom(random).
							setProvider(provider));
			
			encryptor.addMethod(
				new JcePublicKeyKeyEncryptionMethodGenerator(publicKey).
						setProvider(provider).
						setSecureRandom(random));
			
			byte[] bytes = targetOutputStream.toByteArray();
	
			OutputStream wrappedOutputStream = encryptor.open(encryptedStream, bytes.length);
			wrappedOutputStream.write(bytes);
			
			encryptor.close();
			encryptedStream.close();
			
			return tempFile.length();
		} finally {
			tempFileStream.close();
			tempFile.delete();
		}
	}
}