package gr.cite.geoanalytics.dataaccess.entities.project.dao;

import java.util.List;
import java.util.UUID;

import javax.persistence.NoResultException;
import javax.persistence.Query;
import javax.persistence.TypedQuery;

import org.springframework.stereotype.Repository;

import gr.cite.geoanalytics.dataaccess.dao.JpaDao;
import gr.cite.geoanalytics.dataaccess.entities.geocode.GeocodeSystem;
import gr.cite.geoanalytics.dataaccess.entities.layer.Layer;
import gr.cite.geoanalytics.dataaccess.entities.project.Project;
import gr.cite.geoanalytics.dataaccess.entities.project.ProjectLayer;
import gr.cite.geoanalytics.dataaccess.entities.tenant.Tenant;

@Repository
public class ProjectLayerDaoImpl extends JpaDao<ProjectLayer, UUID> implements ProjectLayerDao {

	@Override
	public ProjectLayer find(Project p, Layer l) {
		Query query = entityManager.createQuery("from ProjectLayer pl where pl.layer = :l and pl.project = :p", 
				ProjectLayer.class);
		query.setParameter("l", l);
		query.setParameter("p", p);
		
		try {
			return (ProjectLayer)query.getSingleResult();
		}catch(NoResultException e) {
			return null;
		}
	}

	@Override
	public List<Layer> findByProject(Project p) {
		TypedQuery<Layer> query = 
				entityManager.createQuery("select pl.layer from ProjectLayer pl where pl.project = :p", Layer.class);
		
		query.setParameter("p", p);
		return query.getResultList();
	}
	
	@Override
	public List<GeocodeSystem> findByTenant(Tenant tenant) {
		TypedQuery<GeocodeSystem> query = 
				entityManager.createQuery( //nested query employed because distinct needs to compare all columns and comparison for xml types is not defined
						"select t2 from GeocodeSystem t2 where t2.id in (" +
						"select distinct t.id from ProjectLayer pt " +
						"join pt.term tt join tt.taxonomy t join pt.project p join p.creator u " +
						"where u.tenant = :tenant)",
						GeocodeSystem.class);
		
		query.setParameter("tenant", tenant);
		
		return query.getResultList();
	}
	
	
	@Override
	public Layer findByProjectAndGeocodeSystem(Project p, GeocodeSystem t) {
		TypedQuery<Layer> query = 
				//entityManager.createQuery("select pt.term from ProjectLayer pt join pt.term tt where pt.project = :p and tt.taxonomy = :t", TaxonomyTerm.class);
				entityManager.createQuery("select pl.layer from ProjectLayer pl, Layer l where pl.project = :p and pl.layer = l and l.taxonomy = :t", Layer.class);
		
		query.setParameter("p", p);
		query.setParameter("t", t);
		
		try {
			return query.getSingleResult();
		}catch(NoResultException e) {
			return null;
		}
	}
	
	@Override
	public List<Project> findByLayer(Layer l) {
		TypedQuery<Project> query = 
				entityManager.createQuery("select pl.project from ProjectLayer pl where pl.layer = :l", Project.class);
		
		query.setParameter("l", l);
		return query.getResultList();
	}
	
	@Override
	public void deleteByProject(Project p) {
		Query query = entityManager.createQuery("delete ProjectLayer pt where pt.project = :p");
		query.setParameter("p", p);
		query.executeUpdate();
	}

	@Override
	public void deleteByLayer(Layer l) {
		Query query = entityManager.createQuery("delete ProjectLayer pl where pl.layer = :l");
		query.setParameter("l", l);
		query.executeUpdate();
	}

	@Override
	public ProjectLayer loadDetails(ProjectLayer pl) {
		pl.getCreator().getName();
		pl.getProject().getId();
		pl.getLayer().getId();
		return pl;
	}

}
