package gr.cite.geoanalytics.dataaccess.entities.project.dao;

import java.util.List;
import java.util.UUID;

import javax.persistence.NoResultException;
import javax.persistence.Query;
import javax.persistence.TypedQuery;

import org.springframework.stereotype.Repository;

import gr.cite.geoanalytics.dataaccess.dao.JpaDao;
import gr.cite.geoanalytics.dataaccess.entities.project.Project;
import gr.cite.geoanalytics.dataaccess.entities.project.ProjectTerm;
import gr.cite.geoanalytics.dataaccess.entities.taxonomy.Taxonomy;
import gr.cite.geoanalytics.dataaccess.entities.taxonomy.TaxonomyTerm;
import gr.cite.geoanalytics.dataaccess.entities.tenant.Tenant;

@Repository
public class ProjectTermDaoImpl extends JpaDao<ProjectTerm, UUID> implements ProjectTermDao {

	@Override
	public ProjectTerm find(Project p, TaxonomyTerm tt) {
		Query query = entityManager.createQuery("from ProjectTerm pt where pt.term = :t and pt.project = :p", 
				ProjectTerm.class);
		query.setParameter("t", tt);
		query.setParameter("p", p);
		
		try {
			return (ProjectTerm)query.getSingleResult();
		}catch(NoResultException e) {
			return null;
		}
	}

	@Override
	public List<TaxonomyTerm> findByProject(Project p) {
		TypedQuery<TaxonomyTerm> query = 
				entityManager.createQuery("select pt.term from ProjectTerm pt where pt.project = :p", TaxonomyTerm.class);
		
		query.setParameter("p", p);
		return query.getResultList();
	}
	
	@Override
	public List<Taxonomy> findByTenant(Tenant tenant) {
		TypedQuery<Taxonomy> query = 
				entityManager.createQuery( //nested query employed because distinct needs to compare all columns and comparison for xml types is not defined
						"select t2 from Taxonomy t2 where t2.id in (" +
						"select distinct t.id from ProjectTerm pt " +
						"join pt.term tt join tt.taxonomy t join pt.project p join p.creator u " +
						"where u.tenant = :tenant)",
						Taxonomy.class);
		
		query.setParameter("tenant", tenant);
		
		return query.getResultList();
	}
	
	
	@Override
	public TaxonomyTerm findByProjectAndTaxonomy(Project p, Taxonomy t) {
		TypedQuery<TaxonomyTerm> query = 
				//entityManager.createQuery("select pt.term from ProjectTerm pt join pt.term tt where pt.project = :p and tt.taxonomy = :t", TaxonomyTerm.class);
				entityManager.createQuery("select pt.term from ProjectTerm pt, TaxonomyTerm tt where pt.project = :p and pt.term = tt and tt.taxonomy = :t", TaxonomyTerm.class);
		
		query.setParameter("p", p);
		query.setParameter("t", t);
		
		try {
			return query.getSingleResult();
		}catch(NoResultException e) {
			return null;
		}
	}
	
	@Override
	public List<Project> findByTerm(TaxonomyTerm tt) {
		TypedQuery<Project> query = 
				entityManager.createQuery("select pt.project from ProjectTerm pt where pt.term = :tt", Project.class);
		
		query.setParameter("tt", tt);
		return query.getResultList();
	}
	
	@Override
	public void deleteByProject(Project p) {
		Query query = entityManager.createQuery("delete ProjectTerm pt where pt.project = :p");
		query.setParameter("p", p);
		query.executeUpdate();
	}

	@Override
	public void deleteByTerm(TaxonomyTerm tt) {
		Query query = entityManager.createQuery("delete ProjectTerm pt where pt.term = :tt");
		query.setParameter("tt", tt);
		query.executeUpdate();
	}

	@Override
	public ProjectTerm loadDetails(ProjectTerm pt) {
		pt.getCreator().getName();
		pt.getProject().getId();
		pt.getTerm().getId();
		return pt;
	}

}
