package eu.dnetlib.data.mdstore.modular.mongodb;

import java.util.List;
import java.util.Map;
import javax.xml.stream.XMLStreamException;

import com.google.common.collect.Lists;
import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import com.mongodb.WriteConcern;
import com.mongodb.bulk.BulkWriteResult;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.model.BulkWriteOptions;
import com.mongodb.client.model.ReplaceOneModel;
import com.mongodb.client.model.UpdateOptions;
import com.mongodb.client.model.WriteModel;
import eu.dnetlib.data.mdstore.modular.RecordParser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.bson.BsonSerializationException;

public class MongoBulkWritesManager {

	private static final Log log = LogFactory.getLog(MongoBulkWritesManager.class);
	private final boolean discardRecords;
	private RecordParser recordParser;
	private MongoCollection<DBObject> validCollection;
	private List<WriteModel<DBObject>> validBulkOperationList;
	private BulkWriteOptions writeOptions;
	private MongoCollection<DBObject> discardedCollection;
	private List<WriteModel<DBObject>> discardedBulkOperationList;
	private int discardedOpCounter;
	private int bulkSize;

	public MongoBulkWritesManager(final MongoCollection<DBObject> collection,
			final MongoCollection<DBObject> discardedCollection,
			final int bulkSize,
			final RecordParser parser,
			final boolean discardRecords) {
		this.validCollection = collection.withWriteConcern(WriteConcern.JOURNALED);
		this.validBulkOperationList = Lists.newArrayList();
		this.discardedCollection = discardedCollection.withWriteConcern(WriteConcern.ACKNOWLEDGED);
		this.discardedBulkOperationList = Lists.newArrayList();
		this.discardedOpCounter = 0;
		this.bulkSize = bulkSize;
		this.recordParser = parser;
		this.discardRecords = discardRecords;
		this.writeOptions = new BulkWriteOptions().ordered(false);
	}

	public void insert(final String record) {
		try {
			final Map<String, String> recordProperties = recordParser.parseRecord(record);
			//log.debug("found props: " + recordProperties);
			if (recordProperties.containsKey("id")) {
				if(log.isDebugEnabled()){
					log.debug("Parsed record id "+recordProperties.get("id"));
				}
				final DBObject obj = buildDBObject(record, recordProperties);
//				if (log.isDebugEnabled()) {
//					log.debug("Saving object" + obj);
//				}
				validBulkOperationList.add(new ReplaceOneModel(new BasicDBObject("id", obj.get("id")), obj, new UpdateOptions().upsert(true)));
				if (((validBulkOperationList.size() % bulkSize) == 0) && (!validBulkOperationList.isEmpty())) {
					log.debug("Bulk writing #records: "+validBulkOperationList.size());
					BulkWriteResult res = validCollection.bulkWrite(validBulkOperationList, writeOptions);
					if(log.isDebugEnabled()) {
						log.debug("Inserted: " + res.getInsertedCount());
						if (res.isModifiedCountAvailable()) log.debug("Modified: " + res.getModifiedCount());
					}
					validBulkOperationList = Lists.newArrayList();
				}
			} else {
				if (discardRecords) {
					log.debug("parsed record seems invalid");
					discardRecord(record);
				}
			}
		} catch(BsonSerializationException e){
			log.error("Probably too big XML record: " + e.getMessage(), e);
			log.error("Dropping operation list -- #ops "+validBulkOperationList.size());
			validBulkOperationList = Lists.newArrayList();
		} catch(XMLStreamException e) {
			if (discardRecords) {
				log.error("unhandled exception: " + e.getMessage(), e);
				discardRecord(record);
			}
		}
	}

	private void discardRecord(final String record) {
		discardedCollection.insertOne(new BasicDBObject("body", record));
	}

	public void flushBulks() {
		//setting to journaled write concern to be sure that when the write returns everything has been flushed to disk (https://docs.mongodb.org/manual/faq/developers/#when-does-mongodb-write-updates-to-disk)
		//the explicit fsync command can't be run anymore: 'Command failed with error 13: 'fsync may only be run against the admin database.'
		if (!validBulkOperationList.isEmpty()) {
		//	validCollection = getCollectionWithWriteConcern(validCollection, WriteConcern.JOURNALED);
			validCollection.bulkWrite(validBulkOperationList, writeOptions);
		}

		//setting write concern back to ACKNOWLEDGE to avoid the execution of future writes all in Journaled mode
		//validCollection = getCollectionWithWriteConcern(validCollection, WriteConcern.ACKNOWLEDGED);
		//discardedCollection = getCollectionWithWriteConcern(discardedCollection, WriteConcern.ACKNOWLEDGED);
	}

	private DBObject buildDBObject(final String record, final Map<String, String> recordProperties) {
		final DBObject obj = new BasicDBObject();
		obj.put("id", recordProperties.get("id"));
		obj.put("originalId", recordProperties.get("originalId"));
		obj.put("body", record);
		obj.put("timestamp", System.currentTimeMillis());
		return obj;
	}

	private MongoCollection<DBObject> getCollectionWithWriteConcern(MongoCollection<DBObject> collection, WriteConcern writeConcern) {
		return collection.withWriteConcern(writeConcern);
	}

}
