package eu.dnetlib.openaire.info;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import eu.dnetlib.OpenaireExporterConfig;
import eu.dnetlib.openaire.project.dao.JdbcApiDao;
import eu.dnetlib.openaire.project.dao.ProjectTsvRepository;
import eu.dnetlib.openaire.project.dao.ValueCleaner;
import eu.dnetlib.openaire.project.domain.Project;
import eu.dnetlib.openaire.project.domain.db.ProjectDetails;
import eu.dnetlib.openaire.project.domain.db.ProjectTsv;
import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import javax.sql.DataSource;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.sql.*;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * Created by alessia on 29/04/2020
 *
 * Get and set info dates via JDBC. Dates are expected to be in a table named 'info' with two columns:
 * key - see JdbcInfoDao.DATE_INFO enum
 * value - the date (LocalDate, no time)
 *
 */
@Component
@ConditionalOnProperty(value = "openaire.exporter.enable.info", havingValue = "true")
public class JdbcInfoDaoImpl implements JdbcInfoDao {

	private static final Log log = LogFactory.getLog(JdbcInfoDaoImpl.class);

	@Autowired
	private OpenaireExporterConfig config;

	@Autowired
	private DataSource dataSource;

	@Override
	@Cacheable("info")
	public LocalDate getDate(final DATE_INFO dateInfo) {
		final String sql = "SELECT value FROM info WHERE key=?";

		LocalDate date = null;
		try (final Connection con = getConn(); final PreparedStatement stm = getStm(sql, con, dateInfo.name()); final ResultSet rs = getRs(stm)) {
			log.info("loading info "+dateInfo+" Query: "+stm.toString());
			if(rs.next()) {
				date = rs.getObject("value", LocalDate.class);
			}
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}

		return date;
	}

	@Override
	@CacheEvict(cacheNames = { "info" }, allEntries = true)
	@Scheduled(fixedDelayString = "${openaire.exporter.cache.ttl}")
	public void dropCache() {
		log.debug("dropped info cache");
	}

	private Connection getConn() throws SQLException {
		final Connection connection = dataSource.getConnection();
		connection.setAutoCommit(false);
		return connection;
	}

	private PreparedStatement getStm(final String sql, final Connection con, final String param) throws SQLException {
		final PreparedStatement stm = con.prepareStatement(sql, ResultSet.TYPE_FORWARD_ONLY);
		stm.setString(1, param);
		stm.setFetchSize(config.getJdbc().getMaxRows());
		return stm;
	}

	private ResultSet getRs(final PreparedStatement stm) throws SQLException {
		final ResultSet rs = stm.executeQuery();
		rs.setFetchSize(config.getJdbc().getMaxRows());
		return rs;
	}

}
