package eu.dnetlib.organizations.metrics;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import eu.dnetlib.common.metrics.MetricsCalculator;
import eu.dnetlib.organizations.model.utils.BrowseEntry;
import eu.dnetlib.organizations.model.view.SuggestionInfoViewByCountry;
import eu.dnetlib.organizations.model.view.UserView;
import eu.dnetlib.organizations.repository.OrganizationRepository;
import eu.dnetlib.organizations.repository.readonly.SuggestionInfoViewByCountryRepository;
import eu.dnetlib.organizations.repository.readonly.UserViewRepository;
import eu.dnetlib.organizations.utils.DatabaseUtils;
import eu.dnetlib.organizations.utils.OrganizationStatus;
import io.prometheus.client.Collector.MetricFamilySamples;
import io.prometheus.client.GaugeMetricFamily;

@Component
public class OrganizationMetricsCalculator implements MetricsCalculator {

	@Autowired
	private OrganizationRepository organizationRepository;

	@Autowired
	private UserViewRepository userViewRepository;

	@Autowired
	private DatabaseUtils dbUtils;

	@Autowired
	private SuggestionInfoViewByCountryRepository suggestionInfoViewByCountryRepository;

	@Override
	public List<MetricFamilySamples> getMetrics() {

		final List<MetricFamilySamples> list = new ArrayList<>();

		final OrganizationMetrics m = prepareMetrics(null);

		list.add(new GaugeMetricFamily("openorgs_valid_organizations", "approved organisations", m.getValidOrgs()));
		list.add(new GaugeMetricFamily("openorgs_suggested_organizations", "suggested organisations", m.getPendingOrgs()));
		list.add(new GaugeMetricFamily("openorgs_with_duplicates_orgs", "orgs with new duplicates", m.getOrgsWithSuggestedDuplicates()));
		list.add(new GaugeMetricFamily("openorgs_potential_conflicts", "potential conflicts", m.getPotentialConflicts()));
		list.add(new GaugeMetricFamily("openorgs_national_curators", "Number of curators", m.getNationalCurators()));
		list.add(new GaugeMetricFamily("openorgs_national_curators_countries", "Number of collaborating countries", m.getNationalCuratorCountries()));

		m.getByType()
				.forEach((k, v) -> list.add(new GaugeMetricFamily(
						"openorgs_approved_organizations_type_" + k.toLowerCase(),
						"Number of approved organisations by type (" + k + ")",
						v)));

		return list;

	}

	public OrganizationMetrics prepareMetrics(final String country) {

		long nUsers = 0;
		long nUsersByCountry = 0;

		final Set<String> nUserCountries = new HashSet<>();

		for (final UserView u : userViewRepository.findAll()) {
			if (u.isValid()) {
				nUsers += 1;
				Collections.addAll(nUserCountries, u.getCountries());
			}
			if (country != null && u.isValid() && Arrays.asList(u.getCountries()).contains(country)) {
				nUsersByCountry += 1;
			}
		}
		nUserCountries.remove("UNKNOWN");

		String countryLabel = country;
		long nSuggested = 0;
		long nDups = 0;
		long nConflicts = 0;

		long nSuggestedByCountry = 0;
		long nDupsByCountry = 0;
		long nConflictsByCountry = 0;

		for (final SuggestionInfoViewByCountry info : suggestionInfoViewByCountryRepository.findAll()) {
			nSuggested += info.getnPendingOrgs();
			nDups += info.getnDuplicates();
			nConflicts += info.getnConflicts();
			if (country != null && country.equals(info.getCode())) {
				countryLabel = StringUtils.firstNonBlank(info.getName(), info.getCode(), country);
				nSuggestedByCountry = info.getnPendingOrgs();
				nDupsByCountry = info.getnDuplicates();
				nConflictsByCountry = info.getnConflicts();
			}
		}

		final OrganizationMetrics m = new OrganizationMetrics();
		m.setName("OpenOrgs Metrics");
		m.setImageUrl("openorgs_logo.png");
		m.setValidOrgs(organizationRepository.countByStatus(OrganizationStatus.approved.toString()));
		m.setPendingOrgs(nSuggested);
		m.setOrgsWithSuggestedDuplicates(nDups);
		m.setPotentialConflicts(nConflicts);
		m.setNationalCurators(nUsers);
		m.setNationalCuratorCountries(nUserCountries.size());
		m.setByType(asTerms(dbUtils.browseTypes()));

		if (country != null) {
			final OrganizationMetrics mc = new OrganizationMetrics();

			mc.setName(countryLabel);
			mc.setImageUrl(country + ".gif");

			mc.setValidOrgs(organizationRepository.countByStatusAndCountry(OrganizationStatus.approved.toString(), country));
			mc.setPendingOrgs(nSuggestedByCountry);
			mc.setOrgsWithSuggestedDuplicates(nDupsByCountry);
			mc.setPotentialConflicts(nConflictsByCountry);
			mc.setNationalCurators(nUsersByCountry);
			mc.setByType(asTerms(dbUtils.browseTypesForCountry(country)));

			m.setByCountry(new LinkedHashMap<>());
			m.getByCountry().put(country, mc);
		}

		return m;
	}

	private Map<String, Long> asTerms(final List<BrowseEntry> list) {
		return list.stream().collect(Collectors.toMap(BrowseEntry::getCode, e -> e.getValues().getOrDefault("approved", 0L)));
	}

}
