package eu.dnetlib.validator2.validation.guideline.openaire;

import eu.dnetlib.validator2.result_models.RequirementLevel;
import eu.dnetlib.validator2.result_models.StandardResult;
import eu.dnetlib.validator2.result_models.ValidationResult;
import eu.dnetlib.validator2.validation.guideline.*;
import eu.dnetlib.validator2.validation.utils.ResultUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import java.util.*;
import java.util.stream.Collectors;

import static eu.dnetlib.validator2.validation.guideline.Cardinality.ONE;

public final class FAIR_Literature_GuidelinesV4Profile extends AbstractOpenAireProfile {

    private static final String[] IDENTIFIER_TYPES = {
            "ARK", "arXiv", "bibcode", "DOI", "EAN13", "EISSN", "Handle", "IGSN", "ISBN",
            "ISSN", "ISTC", "LISSN", "ISSNL", "LSID", "PISSN", "PMID", "PURL", "UPC", "URL", "URN", "WOS",
    };

    private static final String[] RELATION_TYPES = {
            "IsCitedBy", "Cites", "IsSupplementTo", "IsSupplementedBy", "IsContinuedBy",
            "Continues", "IsDescribedBy", "Describes", "HasMetadata", "IsMetadataFor", "HasVersion",
            "IsVersionOf", "IsNewVersionOf", "IsPreviousVersionOf", "IsPartOf", "HasPart", "IsReferencedBy",
            "References", "IsDocumentedBy", "Documents", "IsCompiledBy", "Compiles", "IsVariantFormOf",
            "IsOriginalFormOf", "IsIdenticalTo", "IsReviewedBy", "Reviews", "IsDerivedFrom", "IsSourceOf",
            "IsRequiredBy", "Requires"
    };

    private static final String[] RELATED_RESOURCE_GENERAL_TYPES = {
            "Audiovisual", "Collection", "DataPaper", "Dataset", "Event", "Image", "InteractiveResource",
            "Model", "PhysicalObject", "Service", "Software", "Sound", "Text", "Workflow", "Other"
    };

    private static final String[] RESOURCE_IDENTIFIER_TYPES = {
            "ARK", "DOI", "Handle", "PURL", "URL", "URN"
    };

    //    persistent identifier
    private static final ElementSpec F1_01D_SPEC = Builders.
            forMandatoryElement("datacite:identifier", ONE).
            withMandatoryAttribute("identifierType", RESOURCE_IDENTIFIER_TYPES).
            build();


//    //    persistent identifier
//    private static final ElementSpec F1_02D_SPEC = Builders.
//            forMandatoryElement("datacite:identifier", ONE).
//            withMandatoryAttribute("identifierType", IDENTIFIER_TYPES).
//            build();

//    private static final ElementSpec A1_01M_SPEC = Builders.
//            forMandatoryElement("datacite:rights", ONE).
//            withMandatoryAttribute("rightsURI", ACCESS_RIGHTS_URIS).
//            allowedValues(ACCESS_RIGHTS_TYPES).
//            build();

//    // To be implemented
//    private static final ElementSpec A2_01M_SPEC = Builders.
//            forMandatoryElement("subject", ONE_TO_N).
//            withMandatoryAttribute("subjectScheme").
//            withMandatoryAttribute("schemeURI").
//            build();

//    // To be implemented
//    private static final ElementSpec I1_01M_SPEC = Builders.
//            forMandatoryElement("subject", ONE_TO_N).
//            build();

//    // To be implemented
//    // I1_02M Metadata uses semantic resources
//    private static final ElementSpec I1_02M_SPEC = Builders.
//            forMandatoryElement("subject", ONE).
//            build();


    private static final ElementSpec I3_01M_SPEC_1 = Builders.
            forRecommendedRepeatableElement("datacite:relatedIdentifier").
            withMandatoryAttribute("relatedIdentifierType", IDENTIFIER_TYPES).
            withMandatoryAttribute("relationType", RELATION_TYPES).
            //TODO: For following 3 attributes. Need a way to target relationType attribute of current element
            // - Should be used only with relation type (HasMetadata/IsMetadataFor).
                    withOptionalAttribute("relatedMetadataScheme").
            withOptionalAttribute("schemeURI").
            withOptionalAttribute("schemeType").
            withOptionalAttribute("resourceTypeGeneral", RELATED_RESOURCE_GENERAL_TYPES).
            build();

//    private static final ElementSpec I3_01M_SPEC_2 = Builders.
//            forMandatoryElement("creator", ONE_TO_N).
//            withMandatoryAttribute("creatorName").
//            withMandatoryAttribute("nameIdentifier", NAME_IDENTIFIER_SCHEMES).
//            withMandatoryAttribute("nameIdentifierScheme", NAME_IDENTIFIER_SCHEMES).
//            withMandatoryAttribute("schemeURI", NAME_IDENTIFIER_SCHEMES_URIS).
//            withMandatoryAttribute("affiliation", NAME_IDENTIFIER_SCHEMES).
//            build();

    // TO BE IMPLEMENTED

    private static final ElementSpec LICENSE_CONDITION_SPEC = Builders.
            forRecommendedElement("oaire:licenseCondition").
            withMandatoryIfApplicableAttribute("uri", elementIsPresent("oaire:licenseCondition")).
            withMandatoryIfApplicableAttribute("startDate", elementIsPresent("oaire:licenseCondition")).
            build();

    //TODO: weights for guidelines haven't been finalized. They've been given an arbitrary value of 10.
    public static SyntheticGuideline F1_01D = SyntheticGuideline.of("(meta)data are assigned a globally unique and persistent identifier", "F1", "description", "https://w3id.org/fair/principles/latest/F1", "F",
            1, RequirementLevel.MANDATORY, F1_01D_SPEC);

    public static SyntheticGuideline A1_01M = SyntheticGuideline.of("(meta)data are retrievable by their identifier using a standardized communications protocol", "A1", "description", "https://w3id.org/fair/principles/latest/A1", "A",
        1, RequirementLevel.MANDATORY, F1_01D_SPEC);

    public static SyntheticGuideline A1_01_1M = SyntheticGuideline.of("the protocol is open, free, and universally implementable", "A1.1", "description", "https://w3id.org/fair/principles/latest/A1.1", "A",
            1, RequirementLevel.MANDATORY, F1_01D_SPEC);

    //    public static SyntheticGuideline A1_01_2M = SyntheticGuideline.of("the protocol allows for an authentication and authorization procedure, where necessary", "A1.2", "description", "https://w3id.org/fair/principles/latest/A1.2", "A",
    //            10, RequirementLevel.MANDATORY, A1_01_2M_SPEC);

    //    public static SyntheticGuideline A2_01M = SyntheticGuideline.of("metadata are accessible, even when the data are no longer available", "A2", "description", "https://w3id.org/fair/principles/latest/A2", "A",
    //            10, RequirementLevel.MANDATORY, A2_01M_SPEC);

    public static SyntheticGuideline I3_01M_1 = SyntheticGuideline.of("(meta)data include qualified references to other (meta)data", "I3", "description", "https://w3id.org/fair/principles/latest/I3", "I",
            1, RequirementLevel.MANDATORY, I3_01M_SPEC_1);

    public static SyntheticGuideline R1_1_01M = SyntheticGuideline.of("(meta)data are released with a clear and accessible data usage license", "R1.1", "description", "https://w3id.org/fair/principles/latest/R1.1", "R",
            1, RequirementLevel.MANDATORY, LICENSE_CONDITION_SPEC);

    private static final List<Guideline<Document>> GUIDELINES = Collections.unmodifiableList(
            Arrays.asList(
                    F1_01D,
                    new F2_01M_SPEC_LIT(),
                    new F3_01M_SPEC_LIT(),
                    new F4M_SPEC_LIT(),
                    A1_01M,
                    A1_01_1M,
                    new I1M_SPEC_LIT(),
                    new I2_01M_SPEC_LIT(),
                    I3_01M_1,
                    new R1_01M_SPEC_LIT(),
                    R1_1_01M,
                    new R1_2_01M_SPEC_LIT(),
                    new R1_3_01M_SPEC_LIT()
            )
    );

    private static final Map<String, Guideline> GUIDELINE_MAP = GUIDELINES.stream().
            collect(Collectors.toMap(Guideline::getName, (guideline) -> guideline));

    private static final int MAX_SCORE = GUIDELINES.stream().map(Guideline::getWeight).reduce(0, Integer::sum);

    public FAIR_Literature_GuidelinesV4Profile() {
        super("OpenAIRE FAIR Guidelines for Literature Repositories Profile v4");
    }

    @Override
    public Collection<? extends Guideline<Document>> guidelines() {
        return GUIDELINES;
    }

    /**
     *
     * @param guidelineName
     * @return
     */
    @Override
    public Guideline guideline(String guidelineName) {
        return GUIDELINE_MAP.get(guidelineName);
    }

    @Override
    public int maxScore() {
        return MAX_SCORE;
    }
}


//// TODO this goes to FAIRProfile
//class MetadataCompleteness extends AbstractGuideline<Document> {
//
//    public MetadataCompleteness() {
//        super("MetadataCompleteness", 40);
//    }
//
//    @Override
//    public StandardResult validate(String id, Document t) {
//        DataArchiveGuidelinesV2Profile profile = new DataArchiveGuidelinesV2Profile();
//
//        // <></>ODO: iterate over results and build one StandardResult
//        try {
////            System.out.println("Processing MetadataCompleteness...");
//            ValidationResult res = profile.validate(id, t);
//            Map<String, Result> results = res.results();
//            int MaxScoreMetadataCompleteness = (int) ((res.score()*getWeight())/100);
//
////            System.out.println("Max score DataValidator(%): " + res.score());
////            System.out.println("Weight FAIRG: " + getWeight());
////            System.out.println("Max score MetadataCompleteness: " + MaxScoreMetadataCompleteness);
////            System.out.println("\n\n\n\n");
//
////            for (Map.Entry entry : results.entrySet()) {
////                System.out.println(entry.getKey() + " = " + entry.getValue());
////            }
////            System.out.println(score);
//            return getResult(MaxScoreMetadataCompleteness);
//
////            System.out.println(tempp.status() + " - " + tempp.score());
////            String printout = results.entrySet().stream().
////                    map(entry -> entry.getValue() + ": " + entry.getKey()).collect(Collectors.joining("\n"));
////            System.out.println(printout);
//
////            System.out.println("\n\n\n\n");
//        } catch (Exception e) {
//            System.out.println(e.getMessage());
//            System.out.println(e);
//            e.printStackTrace();
//        }
//
//        return null;
//    }
//
//    private static StandardResult getResult(int score) {
//        String aa;
//        aa = (score > 0) ? "SUCCESS" : "FAILURE";
//        return new Result() {
//            @Override
//            public int score() {
//                return score;
//            }
//
//            @Override
//            public Status status() {
////                    return null;
//                return Status.valueOf(aa);
//            }
//
//            @Override
//            public Iterable<String> warnings() { return null; }
//
//            @Override
//            public Iterable<String> errors() {
//                return null;
//            }
//
//            @Override
//            public String internalError() {
//                return null;
//            }
//        };
//    }
//}

class F2_01M_SPEC_LIT extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(F2_01M_SPEC.class);

    public F2_01M_SPEC_LIT() {
        super("data are described with rich metadata", "F2", "description", "https://w3id.org/fair/principles/latest/F2", "F", 1, RequirementLevel.MANDATORY);
    }

    @Override
    public StandardResult validate(String id, Document t) {
        F2_01M_LIT profile = new F2_01M_LIT();

        // TODO: iterate over results and build one StandardResult
        try {
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();

//            Get actual score and not (%) to incorporate to FAIR score
            final int MaxScoreF2_01M_LIT_SPEC = (int) ((res_F.getScore() * getWeight())/100);

//            logger.debug("Max score DataValidator(%): " + res_F.getScore());
//            logger.debug("Weight FAIRG: " + getWeight());
//            logger.debug("Max score F2_01M_SPEC: " + MaxScoreF2_01M_LIT_SPEC);

            final StandardResult ress = getResult(results.entrySet(), MaxScoreF2_01M_LIT_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
            return null;
        }
    }


    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class F3_01M_SPEC_LIT extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(F3_01M_SPEC.class);


    public F3_01M_SPEC_LIT() { super("metadata clearly and explicitly include the identifier of the data it describes", "F3", "description", "https://w3id.org/fair/principles/latest/F3", "F", 1, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        F3_01M_LIT profile = new F3_01M_LIT();

        // TODO: iterate over results and build one StandardResult
        try {
//            System.out.println("\nMetadata includes the identifier for the data");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
//            int MaxScoreF3_01M_LIT_SPEC = (int) ((res_F.getScore()*getWeight())/100);

            int MaxScoreF3_01M_LIT_SPEC;
            if ((int) res_F.getScore() == 50 )
                MaxScoreF3_01M_LIT_SPEC = (int) (((2*res_F.getScore()) * getWeight())/100);
            else
                MaxScoreF3_01M_LIT_SPEC = (int) ((res_F.getScore() * getWeight())/100);

            final StandardResult ress = getResult(results.entrySet(), MaxScoreF3_01M_LIT_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
            return null;
        }
    }
    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class I2_01M_SPEC_LIT extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(I2_01M_SPEC.class);

    public I2_01M_SPEC_LIT() { super("(meta)data use vocabularies that follow FAIR principles", "I2", "description", "https://w3id.org/fair/principles/latest/I2", "I2", 1, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        I2_01M_LIT profile = new I2_01M_LIT();

        // TODO: iterate over results and build one StandardResult
        try {
            logger.debug("Metadata uses FAIR-compliant vocabularies");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
//            int MaxScoreI2_01M_LIT_SPEC = (int) ((res_F.score()*getWeight())/100);

            int MaxScoreI2_01M_LIT_SPEC;
            if ((int) res_F.getScore() == 50 )
                MaxScoreI2_01M_LIT_SPEC = (int) (((2*res_F.getScore()) * getWeight())/100);
            else
                MaxScoreI2_01M_LIT_SPEC = (int) ((res_F.getScore() * getWeight())/100);

            final StandardResult ress = getResult(results.entrySet(), MaxScoreI2_01M_LIT_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
        }
        return null;
    }

    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class R1_01M_SPEC_LIT extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(R1_01M_SPEC_LIT.class);


    public R1_01M_SPEC_LIT() { super("meta(data) are richly described with a plurality of accurate and relevant attributes", "R1", "description", "https://w3id.org/fair/principles/latest/R1", "R", 1, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        R1_01M_LIT profile = new R1_01M_LIT();

        // TODO: iterate over results and build one StandardResult
        try {
//            System.out.println("\nPlurality of accurate and relevant attributes are provided to allow reuse");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
            int MaxScoreR1_01M_SPEC = (int) ((res_F.getScore() * getWeight())/100);

            final StandardResult ress = getResult(results.entrySet(), MaxScoreR1_01M_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
        }
        return null;
    }

    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class R1_2_01M_SPEC_LIT extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(R1_2_01M_SPEC_LIT.class);


    public R1_2_01M_SPEC_LIT() { super("(meta)data are associated with detailed provenance", "R1.2", "description", "https://w3id.org/fair/principles/latest/R1.2", "R", 1, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        R1_2_01M_LIT profile = new R1_2_01M_LIT();

        // TODO: iterate over results and build one StandardResult
        try {
//            System.out.println("\nMetadata includes provenance information according to a cross-community language");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
            int MaxScoreR1_2_01M_SPEC = (int) ((res_F.getScore() * getWeight())/100);

            final StandardResult ress = getResult(results.entrySet(), MaxScoreR1_2_01M_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
        }
        return null;
    }

    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class R1_3_01M_SPEC_LIT extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(R1_3_01M_SPEC_LIT.class);


    public R1_3_01M_SPEC_LIT() { super("(meta)data meet domain-relevant community standards", "R1.3", "description", "https://w3id.org/fair/principles/latest/R1.3", "R", 1, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        LiteratureGuidelinesV4Profile profile = new LiteratureGuidelinesV4Profile();

        // TODO: iterate over results and build one Guideline.Result
        try {
//            System.out.println("\nMetadata includes provenance information according to a cross-community language");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
            int MaxScoreR1_3_01M_SPEC = (int) ((res_F.getScore() * getWeight())/100);

            final StandardResult ress = getResult(results.entrySet(), MaxScoreR1_3_01M_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
        }
        return null;
    }

    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class F4M_SPEC_LIT extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(F4M_SPEC_LIT.class);


    public F4M_SPEC_LIT() { super("(meta)data are registered or indexed in a searchable resource", "F4", "description", "https://w3id.org/fair/principles/latest/F4", "F", 1, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        LiteratureGuidelinesV4Profile profile = new LiteratureGuidelinesV4Profile();

        // TODO: iterate over results and build one Guideline.Result
        try {
//            System.out.println("\nMetadata includes provenance information according to a cross-community language");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
            int MaxScoreF4M_SPEC = (int) ((res_F.getScore()*getWeight())/100);

            final StandardResult ress = getResult(results.entrySet(), MaxScoreF4M_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
        }
        return null;
    }

    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class I1M_SPEC_LIT extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(I1M_SPEC_LIT.class);


    public I1M_SPEC_LIT() { super("(meta)data use a formal, accessible, shared, and broadly applicable language for knowledge representation", "I1", "description", "https://w3id.org/fair/principles/latest/I1", "I", 1, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        LiteratureGuidelinesV4Profile profile = new LiteratureGuidelinesV4Profile();

        // TODO: iterate over results and build one Guideline.Result
        try {
//            System.out.println("\nMetadata includes provenance information according to a cross-community language");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
            int MaxScoreF4M_SPEC = (int) ((res_F.getScore()*getWeight())/100);

            final StandardResult ress = getResult(results.entrySet(), MaxScoreF4M_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
        }
        return null;
    }

    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}
