package eu.dnetlib.validator2.engine.builtins;

import eu.dnetlib.validator2.engine.RuleValidationResult;
import eu.dnetlib.validator2.engine.TestResultPredicate;
import eu.dnetlib.validator2.engine.contexts.TermsProperty;
import eu.dnetlib.validator2.engine.contexts.XMLContextWithVocabulary;
import org.w3c.dom.NodeList;

import java.util.Map;

public class XMLVocabularyRule extends XMLRule<XMLContextWithVocabulary> {

    protected XMLVocabularyRule(XMLContextWithVocabulary context) {
        super(context, (NodeList nodes) -> {
            TermsProperty terms = context.getTermsProperty();
            String termsType = context.getTermsTypeProperty().getValue().toLowerCase();

            TestResultPredicate<String> termExistsPredicate = (String value) -> {
                if (terms.termExists(value).isSuccess())
                    return RuleValidationResult.success();
                else
                    return RuleValidationResult.failure("value '" + value + "' not found in vocabulary", "Use a value from the controlled vocabulary: " + terms.getValue());
            };

            // The NodeListActionProperty.test method now expects a TestResultPredicate
            RuleValidationResult result = context.getNodeListActionProperty().test(nodes, termExistsPredicate);

            //TODO: Check original implementation for blacklist/whitelist. It looks weird.
            if ( termsType.equals("blacklist") ) {
                // If the predicate succeeded, it means the value was found in the vocabulary (blacklist)
                if ( result.isSuccess() )
                    return RuleValidationResult.failure("value found in blacklist: " + result.getMessage(), "Remove the value or use a value that is not in the blacklist.");
                else
                    // If the predicate failed, it means the value was NOT found in the vocabulary (blacklist), which is a success for blacklist
                    return RuleValidationResult.success();
            } else { // whitelist
                // For whitelist, the result from the predicate is directly the success/failure
                return result;
            }
        });
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder extends AbstractRuleBuilder<XMLVocabularyRule, XMLContextWithVocabulary> {

        private Builder() {
            super(new StandardXMLContextWithVocabulary());
        }

        public Builder setId(String id) {
            context.getIdProperty().setValue(id);
            return this;
        }

        public Builder setXPathExpression(String xpath) {
            context.getXPathExpressionProperty().setValue(xpath);
            return this;
        }

        public Builder setNodeListAction(String nodeListAction) throws RuntimeException {
            context.getNodeListActionProperty().setValue(nodeListAction);
            return this;
        }

        public Builder setVocabularyTerms(String terms) {
            context.getTermsProperty().setValue(terms);
            return this;
        }

        public Builder setVocabularyTermsAndTermsType(String terms, String termsType) {
            context.getTermsProperty().setValue(terms);
            context.getTermsTypeProperty().setValue(termsType);
            return this;
        }

        private void initializeOptionalTermsTypeProperty() {
            if (context.getTermsTypeProperty().getValue() == null) {
                context.getTermsTypeProperty().setValue("blacklist");
            }
        }

        public XMLVocabularyRule build() {
            initializeOptionalTermsTypeProperty();
            ensureContextIsValid();
            return new XMLVocabularyRule(context);
        }

        @Override
        public XMLVocabularyRule buildFrom(Map<String, String> map) {
            context.readFrom(map);
            initializeOptionalTermsTypeProperty();
            ensureContextIsValid();
            return new XMLVocabularyRule(context);
        }
    }

}
