package eu.dnetlib.app.directindex.mapping;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.common.SolrInputDocument;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.fasterxml.jackson.core.JsonProcessingException;

import eu.dnetlib.app.directindex.clients.CommunityClient;
import eu.dnetlib.app.directindex.clients.CommunityClient.ContextInfo;
import eu.dnetlib.app.directindex.clients.DatasourceManagerClient;
import eu.dnetlib.app.directindex.clients.ProjectClient;
import eu.dnetlib.app.directindex.clients.ProjectClient.ProjectInfo;
import eu.dnetlib.app.directindex.clients.VocabularyClient;
import eu.dnetlib.app.directindex.errors.DirectIndexApiException;
import eu.dnetlib.app.directindex.input.DatasourceEntry;
import eu.dnetlib.app.directindex.input.PidEntry;
import eu.dnetlib.app.directindex.input.ResultEntry;
import eu.dnetlib.dhp.schema.common.ModelConstants;
import eu.dnetlib.dhp.schema.solr.AccessRight;
import eu.dnetlib.dhp.schema.solr.Author;
import eu.dnetlib.dhp.schema.solr.BestAccessRight;
import eu.dnetlib.dhp.schema.solr.Category;
import eu.dnetlib.dhp.schema.solr.Concept;
import eu.dnetlib.dhp.schema.solr.Context;
import eu.dnetlib.dhp.schema.solr.Country;
import eu.dnetlib.dhp.schema.solr.Funder;
import eu.dnetlib.dhp.schema.solr.Funding;
import eu.dnetlib.dhp.schema.solr.FundingLevel;
import eu.dnetlib.dhp.schema.solr.Instance;
import eu.dnetlib.dhp.schema.solr.Language;
import eu.dnetlib.dhp.schema.solr.Pid;
import eu.dnetlib.dhp.schema.solr.Provenance;
import eu.dnetlib.dhp.schema.solr.RecordType;
import eu.dnetlib.dhp.schema.solr.RelatedRecord;
import eu.dnetlib.dhp.schema.solr.RelatedRecordHeader;
import eu.dnetlib.dhp.schema.solr.Result;
import eu.dnetlib.dhp.schema.solr.SolrRecord;
import eu.dnetlib.dhp.schema.solr.SolrRecordHeader;
import eu.dnetlib.dhp.schema.solr.SolrRecordHeader.Status;
import eu.dnetlib.dhp.solr.mapping.SolrInputDocumentMapper;

@Component
public class SolrRecordMapper {

	@Autowired
	private DatasourceManagerClient dsmClient;

	@Autowired
	private VocabularyClient vocClient;

	@Autowired
	private CommunityClient communityClient;

	@Autowired
	private ProjectClient projectClient;

	private static final Log log = LogFactory.getLog(SolrRecordMapper.class);

	public SolrInputDocument toSolrInputRecord(final ResultEntry result) throws JsonProcessingException {
		final SolrRecord sr = toSolrRecord(result);
		return SolrInputDocumentMapper.map(sr, XMLSolrSerializer.generateXML(sr), false);
	}

	public SolrRecord toSolrRecord(final ResultEntry result) {
		final Provenance collectedFrom = prepareProvenance(result.getCollectedFromId());
		final Provenance hostedBy = StringUtils.isNotBlank(result.getHostedById()) ? prepareProvenance(result.getHostedById()) : collectedFrom;

		final List<Pid> pids = result.getPids() == null ? Arrays.asList()
				: result.getPids()
						.stream()
						.map(this::prepareDnetPid)
						.toList();

		final SolrRecord sr = new SolrRecord();

		sr.setHeader(prepareDnetHeader(result));
		sr.setResult(prepareDnetResult(result, collectedFrom, hostedBy, pids));

		// sr.setDatasource(null);
		// sr.setProject(null);
		// sr.setOrganization(null);
		// sr.setPerson(null);

		sr.setCollectedfrom(Arrays.asList(collectedFrom));

		if (result.getLinksToProjects() != null) {
			sr.setLinks(result.getLinksToProjects()
					.stream()
					.map(this::prepareDnetProjectLink)
					.toList());
		}

		sr.setPid(pids);

		if (result.getContexts() != null) {
			sr.setContext(prepareDnetContext(result.getContexts()));
		}

		// sr.setMeasures(null);

		return sr;
	}

	public ResultEntry toResultEntry(final SolrRecord r) {
		if (r == null) { return null; }

		final Result result = r.getResult();
		if (result == null) { return null; }

		final SolrRecordHeader header = r.getHeader();
		if (header == null) { return null; }

		final ResultEntry re = new ResultEntry();

		if (header.getId() != null) {
			re.setOpenaireId(header.getId());
		}

		if (header.getOriginalId() != null) {
			re.setOriginalId(header.getOriginalId().stream().findFirst().orElse(null));
		}

		if (result.getMaintitle() != null) {
			re.setTitle(result.getMaintitle());
		}

		if (result.getAuthor() != null) {
			re.setAuthors(result.getAuthor().stream().map(Author::getFullname).toList());
		}

		if (result.getPublisher() != null) {
			re.setPublisher(result.getPublisher());
		}

		if (result.getDescription() != null) {
			re.setDescription(result.getDescription().stream().findFirst().orElse(null));
		}

		if (result.getLanguage() != null) {
			re.setLanguage(result.getLanguage().getCode());
		}

		if (header.getRecordType() != null) {
			re.setType(header.getRecordType().toString());
		}

		if (result.getEmbargoenddate() != null) {
			re.setEmbargoEndDate(result.getEmbargoenddate());
		}

		if (result.getBestaccessright() != null) {
			re.setAccessRightCode(result.getBestaccessright().getCode());
		}

		if (r.getPid() != null) {
			re.setPids(r.getPid().stream().map(p -> new PidEntry(p.getTypeCode(), p.getValue())).toList());
		}

		final Instance instance = result.getInstance().stream().findFirst().orElse(null);
		if (instance != null) {
			re.setResourceType(instance.getInstancetype());
			if (instance.getUrl() != null) {
				re.setUrl(instance.getUrl().stream().findFirst().orElse(null));
			}
			if (instance.getCollectedfrom() != null) {
				re.setCollectedFromId(instance.getCollectedfrom().getDsId());
			}
			if (instance.getHostedby() != null) {
				re.setHostedById(instance.getHostedby().getDsId());
			}
		}

		final List<String> contexts = new ArrayList<String>();
		if (r.getContext() != null) {
			for (final Context ctx : r.getContext()) {
				if (ctx.getCategory() != null && ctx.getCategory().size() > 0) {
					for (final Category cat : ctx.getCategory()) {
						if (cat.getConcept() != null && cat.getConcept().size() > 0) {
							for (final Concept cpt : cat.getConcept()) {
								contexts.add(cpt.getId());
							}
						}
					}
				} else {
					contexts.add(ctx.getId());
				}
			}
		}
		re.setContexts(contexts);

		// @formatter:off
		if (r.getLinks() != null) {
			final List<String> projects = r.getLinks()
				.stream()
				.filter(l -> l.getHeader().getRelatedRecordType() == RecordType.project)
				.filter(p -> p.getFunding() != null)
				.filter(p -> p.getFunding().getFunder() != null)
				.filter(p -> p.getFunding().getLevel0() != null)
				.filter(p -> p.getFunding().getFunder().getJurisdiction() != null)
				.map(p -> String.format("info:eu-repo/grantAgreement/%s/%s/%s/%s/%s/%s",
						StringUtils.defaultIfBlank(p.getFunding().getFunder().getShortname(), ""),
						StringUtils.defaultIfBlank(p.getFunding().getLevel0().getName(), ""),
						StringUtils.defaultIfBlank(p.getCode(), ""),
						StringUtils.defaultIfBlank(p.getFunding().getFunder().getJurisdiction().getCode(), ""),
						StringUtils.defaultIfBlank(p.getProjectTitle(), ""),
						StringUtils.defaultIfBlank(p.getAcronym(), "")))
				.distinct()
				.collect(Collectors.toList());

			re.setLinksToProjects(projects);
		}
		// @formatter:on

		return re;
	}

	private SolrRecordHeader prepareDnetHeader(final ResultEntry re) {
		final SolrRecordHeader header = new SolrRecordHeader();
		header.setId(re.getOpenaireId());
		header.setOriginalId(Arrays.asList(re.getOriginalId()));
		header.setRecordType(StringUtils.isNotBlank(re.getType()) ? RecordType.valueOf(re.getType()) : RecordType.publication);
		header.setStatus(Status.UNDER_CURATION);
		header.setDeletedbyinference(false);
		return header;
	}

	@SuppressWarnings("deprecation")
	private Result prepareDnetResult(final ResultEntry re, final Provenance collectedFrom, final Provenance hostedBy, final List<Pid> pids) {
		final Result r = new Result();

		// String getAnyId() {

		r.setResulttype(StringUtils.firstNonBlank(re.getType(), "publication"));
		r.setMaintitle(re.getTitle());

		if (re.getAuthors() != null) {
			final List<Author> list = new ArrayList<Author>();
			for (int i = 0; i < re.getAuthors().size(); i++) {
				list.add(Author.newInstance(re.getAuthors().get(i), null, null, i, Arrays.asList()));
			}
			r.setAuthor(list);
		}

		if (StringUtils.isNotBlank(re.getDescription())) {
			r.setDescription(Arrays.asList(re.getDescription()));
		}

		if (StringUtils.isNotBlank(re.getLanguage())) {
			final String code = re.getLanguage();
			final String label = vocClient.findTermLabel("dnet:languages", re.getLanguage());
			r.setLanguage(Language.newInstance(code, label));
		}

		if (StringUtils.isNotBlank(re.getPublisher())) {
			r.setPublisher(re.getPublisher());
		}

		if (StringUtils.isNotBlank(re.getEmbargoEndDate())) {
			r.setEmbargoenddate(re.getEmbargoEndDate());
		}

		if (StringUtils.isNotBlank(re.getAccessRightCode())) {
			final String code = re.getAccessRightCode();
			final String label = vocClient.findTermLabel("dnet:access_modes", re.getAccessRightCode());
			r.setBestaccessright(BestAccessRight.newInstance(code, label));
		} else {
			r.setBestaccessright(BestAccessRight.newInstance("UNKNOWN", "not available"));
		}

		// TODO: To replace with Instance after the dismission of the XML
		final InstanceWithTypeCode instance = new InstanceWithTypeCode();
		instance.setInstancetype(vocClient.findTermLabel("dnet:publication_resource", re.getResourceType()));
		instance.setInstancetypeCode(re.getResourceType());
		instance.setCollectedfrom(collectedFrom);
		instance.setHostedby(hostedBy);
		instance.setPid(pids);

		if (StringUtils.isNotBlank(re.getUrl())) {
			instance.setUrl(Arrays.asList(re.getUrl()));
		}

		if (StringUtils.isNotBlank(re.getAccessRightCode())) {
			final String code = re.getAccessRightCode();
			final String label = vocClient.findTermLabel("dnet:access_modes", re.getAccessRightCode());
			instance.setAccessright(AccessRight.newInstance(code, label, null));
		} else {
			instance.setAccessright(AccessRight.newInstance("UNKNOWN", "not available", null));
		}

		r.setInstance(Arrays.asList(instance));

		return r;
	}

	private Provenance prepareProvenance(final String dsId) {
		final Provenance p = new Provenance();
		p.setDsId(dsId);
		try {
			p.setDsName(dsmClient.findDatasource(dsId).getName());
		} catch (final DirectIndexApiException e) {
			log.warn("Invalid datasource id: " + dsId, e);
			p.setDsName("UNRECOGNIZED DATASOURCE");
		}
		return p;
	}

	private RelatedRecord prepareDnetProjectLink(final String link) {

		// info:eu-repo/grantAgreement/EC/FP7/244909/EU/Making Capabilities Work/WorkAble

		final ProjectInfo info = projectClient.resolveProjectLink(link);

		if (info == null) { return null; }

		final RelatedRecordHeader head = new RelatedRecordHeader();
		head.setRelatedIdentifier(info.getId());
		head.setRelationType(ModelConstants.RESULT_PROJECT);
		head.setRelatedRecordType(RecordType.project);
		head.setRelationClass(ModelConstants.IS_PRODUCED_BY);
		head.setRelationProvenance(ModelConstants.USER_CLAIM);
		head.setTrust("0.9");

		final RelatedRecord rel = new RelatedRecord();
		rel.setHeader(head);
		rel.setProjectTitle(info.getTitle());
		rel.setAcronym(info.getAcronym());
		rel.setCode(info.getCode());

		if (StringUtils.isNotBlank(info.getFunderId())) {
			final Funding funding = new Funding();

			final Funder funder = new Funder();
			funder.setId(info.getFunderId());
			funder.setName(info.getFunderName());
			funder.setShortname(info.getFunderShortName());
			funder.setJurisdiction(Country.newInstance(info.getJurisdiction(), vocClient.findTermLabel("dnet:countries", info.getJurisdiction())));

			funding.setFunder(funder);
			if (StringUtils.isNotBlank(info.getFundingId())) {
				final FundingLevel level = new FundingLevel();
				level.setId(info.getFundingId());
				level.setName(info.getFundingName());
				level.setDescription(info.getFundingName());
				funding.setLevel0(level);
			}

			rel.setFunding(funding);
		}

		return rel;
	}

	private Pid prepareDnetPid(final PidEntry pe) {
		final Pid p = new Pid();
		p.setValue(pe.getValue());
		p.setTypeCode(pe.getType());
		p.setTypeLabel(vocClient.findTermLabel("dnet:pid_types", pe.getType()));
		return p;
	}

	private List<Context> prepareDnetContext(final List<String> list) {

		final Map<String, ContextInfo> nodes = new HashMap<>();

		for (final String id : list) {
			if (id.contains(CommunityClient.ZENODO_COMMUNITY)) {
				final String zenodoId = calculateZenodoId(id);

				for (final String path : communityClient.findOpenaireCommunities(zenodoId)) {
					nodes.putAll(communityClient.findNodes(path));
				}
			} else {
				nodes.putAll(communityClient.findNodes(id));
			}
		}

		final List<Context> res = new ArrayList<Context>();
		filterContextInfoByLevel(nodes, 0, null).forEach((k, v) -> {
			res.add(Context.newInstance(k, v.getLabel(), v.getType(), prepareDnetCategories(nodes, k)));
		});

		return res;

	}

	protected String calculateZenodoId(final String context) {
		if (StringUtils.isBlank(context)) { return null; }

		final String[] arr = StringUtils.split(context, "/");

		for (int i = arr.length - 1; i >= 0; i--) {
			if (StringUtils.isNotBlank(arr[i])) { return arr[i]; }
		}

		return null;
	}

	private List<Category> prepareDnetCategories(final Map<String, ContextInfo> nodes, final String parent) {
		final List<Category> res = new ArrayList<Category>();

		filterContextInfoByLevel(nodes, 1, parent).forEach((k, v) -> {
			final Category cat = Category.newInstance(k, v.getLabel());
			cat.setConcept(prepareDnetConcepts(nodes, k, 2));
			res.add(cat);
		});

		return res;
	}

	private List<Concept> prepareDnetConcepts(final Map<String, ContextInfo> nodes, final String parent, final int level) {
		final List<Concept> res = new ArrayList<Concept>();

		filterContextInfoByLevel(nodes, level, parent).forEach((k, v) -> {
			res.add(Concept.newInstance(k, v.getLabel()));
		});

		return res;
	}

	private Map<String, ContextInfo> filterContextInfoByLevel(final Map<String, ContextInfo> nodes, final int level, final String parent) {
		return nodes.entrySet()
				.stream()
				.filter(e -> parent == null || e.getKey().startsWith(parent + "::"))
				.filter(e -> e.getKey().split("::").length == level)
				.collect(Collectors.toMap(Entry::getKey, Entry::getValue));
	}

	public String calculateOpenaireId(final String originalId, final String collectedFromId) throws DirectIndexApiException {
		return calculateOpenaireId(originalId, dsmClient.findDatasource(collectedFromId));
	}

	private String calculateOpenaireId(final String originalId, final DatasourceEntry collectedFromEntry) {
		return collectedFromEntry.getPrefix() + "::" + DigestUtils.md5Hex(originalId);
	}

	public DatasourceManagerClient getDsmClient() {
		return dsmClient;
	}

	public void setDsmClient(final DatasourceManagerClient dsmClient) {
		this.dsmClient = dsmClient;
	}

	public VocabularyClient getVocClient() {
		return vocClient;
	}

	public void setVocClient(final VocabularyClient vocClient) {
		this.vocClient = vocClient;
	}

	public CommunityClient getCommunityClient() {
		return communityClient;
	}

	public void setCommunityClient(final CommunityClient communityClient) {
		this.communityClient = communityClient;
	}

	public ProjectClient getProjectClient() {
		return projectClient;
	}

	public void setProjectClient(final ProjectClient projectClient) {
		this.projectClient = projectClient;
	}

}
