package eu.dnetlib.app.directindex.controllers;

import java.util.LinkedHashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import eu.dnetlib.app.directindex.errors.DirectIndexApiException;
import eu.dnetlib.app.directindex.input.ResultEntry;
import eu.dnetlib.app.directindex.mapping.SolrRecordMapper;
import eu.dnetlib.app.directindex.service.DirectIndexService;
import eu.dnetlib.app.directindex.solr.SolrIndexClient;
import eu.dnetlib.dhp.schema.solr.SolrRecord;

@RestController
@RequestMapping("/api/results")
@ConditionalOnProperty(value = "dnet.directindex.legacy.enabled", havingValue = "true")
public class LegacyApiController {

	private static final Log log = LogFactory.getLog(LegacyApiController.class);

	@Autowired
	private DirectIndexService service;

	@Autowired
	private SolrRecordMapper solrRecordMapper;

	@Autowired
	private SolrIndexClient solrIndexClient;

	@PostMapping("/feedObject")
	@Deprecated
	public String feedResult_deprecated(@RequestBody final ResultEntry pub, final HttpServletRequest req)
			throws DirectIndexApiException {

		return feedResult(pub, req);
	}

	@PostMapping
	public String feedResult(@RequestBody final ResultEntry pub, final HttpServletRequest req)
			throws DirectIndexApiException {

		return service.prepareMetadataInsertOrUpdate(pub, req.getRemoteAddr());
	}

	@GetMapping("/{openaireId}")
	public Map<String, Object> getResultWithOpenaireId(@PathVariable(value = "openaireId") final String openaireId) throws DirectIndexApiException {

		try {
			final SolrRecord indexed = solrIndexClient.findRecord(openaireId);

			final Map<String, Object> res = new LinkedHashMap<String, Object>();
			res.put("indexed_oaf", indexed);
			res.put("indexed_simple", indexed != null ? solrRecordMapper.toResultEntry(indexed) : null);
			res.put("pending_action", service.findPendingAction(openaireId).orElse(null));

			return res;
		} catch (final Throwable e) {
			throw new DirectIndexApiException("Error searching record with openaireId = " + openaireId, e);
		}
	}

	@DeleteMapping("/{openaireId}")
	public boolean deleteResultWithOpenaireId(@PathVariable(value = "openaireId") final String openaireId, final HttpServletRequest req)
			throws DirectIndexApiException {

		service.prepareMetadataDeletion(openaireId, req.getRemoteAddr());

		return true;
	}

	@DeleteMapping
	public boolean deleteResultWithOriginalId(
			@RequestParam(value = "originalId", required = true) final String originalId,
			@RequestParam(value = "collectedFromId", required = true) final String collectedFromId,
			final HttpServletRequest req) throws DirectIndexApiException {

		final String openaireId = solrRecordMapper.calculateOpenaireId(originalId, collectedFromId);

		service.prepareMetadataDeletion(openaireId, req.getRemoteAddr());

		return true;
	}

	@ExceptionHandler(Exception.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	public @ResponseBody ErrorMessage handleException(final Exception e) {
		log.error("Error in direct index API", e);
		return new ErrorMessage(e);
	}

	public class ErrorMessage {

		private final String message;
		private final String stacktrace;

		public ErrorMessage(final Exception e) {
			this(e.getMessage(), ExceptionUtils.getStackTrace(e));
		}

		public ErrorMessage(final String message, final String stacktrace) {
			this.message = message;
			this.stacktrace = stacktrace;
		}

		public String getMessage() {
			return message;
		}

		public String getStacktrace() {
			return stacktrace;
		}

	}

}
