package gr.uoa.di.driver.enabling.islookup;

import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBException;

import org.apache.log4j.Logger;

import eu.dnetlib.api.enabling.ISLookUpService;
import eu.dnetlib.api.enabling.ISLookUpServiceException;
import eu.dnetlib.domain.DriverResource;
import eu.dnetlib.domain.EPR;
import eu.dnetlib.domain.SearchCriteria;
import gr.uoa.di.driver.enabling.ISLookUp;
import gr.uoa.di.driver.enabling.ISLookUpException;
import gr.uoa.di.driver.enabling.resultset.ResultSet;
import gr.uoa.di.driver.enabling.resultset.ResultSetFactory;
import gr.uoa.di.driver.util.ServiceLocator;
import gr.uoa.di.driver.xml.ResourceToXmlConverter;

public class ISLookUpImpl<D extends DriverResource> implements ISLookUp<D> {
	private static Logger logger = Logger.getLogger(ISLookUpImpl.class);

	private ServiceLocator<ISLookUpService> lookupLocator = null;
	private ResourceToXmlConverter<D> converter = null;
	private ResultSetFactory rsFactory = null;

	public D getById(String profileId) throws ISLookUpException {
		try {
			String xml = lookupLocator.getService().getResourceProfile(
					profileId);

			logger.debug("Collection xml..." + xml);
			
			return this.parseXmlProfile(xml);
		} catch (ISLookUpServiceException e) {
			throw new ISLookUpException(e);
		}
	}

	public List<D> getByid(String[] profileIds) throws ISLookUpException {
		List<D> res = new ArrayList<D>();

		for (String id : profileIds) {
			res.add(this.getById(id));
		}

		return res;
	}

	public List<D> fetch(SearchCriteria criteria) throws ISLookUpException {
		logger.debug("Fetching...");
		try {
			String xQuery = converter.toXQueryString(criteria);
			logger.debug("xQuery: " + xQuery);

			EPR epr = lookupLocator.getService().searchProfile(xQuery);
			ResultSet<String> xmls = rsFactory.createResultSet(epr);
			List<D> res = new ArrayList<D>();

			logger.debug("Getting results from result set. Result set size: "
					+ xmls.size());
			
			logger.debug("starting parsing");
			
			if (xmls.size() > 0) {
				for (String xml : xmls.get(1, xmls.size()))
					res.add(this.parseXmlProfile(xml));
			}
			
			logger.debug("parsed");
			
			return res;
		} catch (ISLookUpServiceException e) {
			logger.error(e);
			throw new ISLookUpException(e);
		}
	}

	private D parseXmlProfile(String xml) throws ISLookUpException {
		try {
			return this.converter.XmlToObject(xml);
		} catch (JAXBException e) {
			throw new ISLookUpException("Error parsing xml profile", e);
		} catch (Exception e) {
			logger.error("Error parsing: " + xml);
			throw new ISLookUpException("Error parsing xml profile", e);
		}
	}

	public List<String> fetch(String XQuery) throws ISLookUpException {
		ResultSet<String> rs;
		try {
			rs = rsFactory.createResultSet(lookupLocator.getService().searchProfile(
					XQuery));

			if (rs.size() > 0)
				return rs.get(1, rs.size());
			else
				return new ArrayList<String>();
		} catch (ISLookUpServiceException e) {
			throw new ISLookUpException(e);
		}
	}

	public List<D> performQuickSearch(SearchCriteria criteria)
			throws ISLookUpException {

		try {
			String xQuery = converter.toXQueryString(criteria);
			List<D> results = new ArrayList<D>();
			List<String> xmls = lookupLocator.getService().quickSearchProfile(
					xQuery);

			for (String profile : xmls)
				results.add(parseXmlProfile(profile));

			return results;
		} catch (ISLookUpServiceException e) {
			throw new ISLookUpException(e);
		}

	}

	public D getUniqueResult(SearchCriteria criteria) throws ISLookUpException {
		List<D> list = this.performQuickSearch(criteria);

		if (list.size() == 0)
			return null;
		else if (list.size() == 1)
			return list.get(0);
		else
			throw new ISLookUpException("More than one results in query");
	}

	@Override
	public EPR search(SearchCriteria criteria) throws ISLookUpException {
		try {
			return lookupLocator.getService().searchProfile(
					converter.toXQueryString(criteria));
		} catch (ISLookUpServiceException e) {
			throw new ISLookUpException(e);
		}
	}

	@Override
	public EPR search(String XQuery) throws ISLookUpException {
		try {
			return lookupLocator.getService().searchProfile(XQuery);
		} catch (ISLookUpServiceException e) {
			throw new ISLookUpException(e);
		}
	}

	public void setLookupLocator(ServiceLocator<ISLookUpService> lookupLocator) {
		this.lookupLocator = lookupLocator;
	}

	public void setConverter(ResourceToXmlConverter<D> converter) {
		this.converter = converter;
	}

	public void setRsFactory(ResultSetFactory rsFactory) {
		this.rsFactory = rsFactory;
	}	
}