package gr.uoa.di.driver.enabling.a2;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * An implementation of the ScheduledExecutorService that takes care of 
 * preserving the value of a ThreadLocal variable in each tasks thread.
 *  
 * @author antleb
 *
 */
public class AASExecutor implements ScheduledExecutorService {
	private ThreadLocal<String> callerSecurityContext = null;
	private ScheduledExecutorService executorService = null;

	@Override
	public ScheduledFuture<?> schedule(Runnable command, long delay,
			TimeUnit unit) {
		return executorService.schedule(new AASRunnable(command), delay, unit);
	}

	@Override
	public <V> ScheduledFuture<V> schedule(Callable<V> callable, long delay,
			TimeUnit unit) {
		return executorService.schedule(new AASCallable<V>(callable), delay,
				unit);
	}

	@Override
	public ScheduledFuture<?> scheduleAtFixedRate(Runnable command,
			long initialDelay, long period, TimeUnit unit) {
		return scheduleAtFixedRate(new AASRunnable(command), initialDelay,
				period, unit);
	}

	@Override
	public ScheduledFuture<?> scheduleWithFixedDelay(Runnable command,
			long initialDelay, long delay, TimeUnit unit) {
		return executorService.scheduleWithFixedDelay(new AASRunnable(command),
				initialDelay, delay, unit);
	}

	@Override
	public boolean awaitTermination(long timeout, TimeUnit unit)
			throws InterruptedException {
		return executorService.awaitTermination(timeout, unit);
	}

	@Override
	public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks)
			throws InterruptedException {
		Collection<Callable<T>> aasTasks = new ArrayList<Callable<T>>();

		for (Callable<T> task : tasks)
			aasTasks.add(new AASCallable<T>(task));

		return executorService.invokeAll(tasks);
	}

	@Override
	public <T> List<Future<T>> invokeAll(
			Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
			throws InterruptedException {
		Collection<Callable<T>> aasTasks = new ArrayList<Callable<T>>();

		for (Callable<T> task : tasks)
			aasTasks.add(new AASCallable<T>(task));

		return executorService.invokeAll(aasTasks, timeout, unit);
	}

	@Override
	public <T> T invokeAny(Collection<? extends Callable<T>> tasks)
			throws InterruptedException, ExecutionException {
		Collection<Callable<T>> aasTasks = new ArrayList<Callable<T>>();

		for (Callable<T> task : tasks)
			aasTasks.add(new AASCallable<T>(task));

		return executorService.invokeAny(aasTasks);
	}

	@Override
	public <T> T invokeAny(Collection<? extends Callable<T>> tasks,
			long timeout, TimeUnit unit) throws InterruptedException,
			ExecutionException, TimeoutException {
		Collection<Callable<T>> aasTasks = new ArrayList<Callable<T>>();

		for (Callable<T> task : tasks)
			aasTasks.add(new AASCallable<T>(task));

		return executorService.invokeAny(aasTasks, timeout, unit);
	}

	@Override
	public boolean isShutdown() {
		return executorService.isShutdown();
	}

	@Override
	public boolean isTerminated() {
		return executorService.isTerminated();
	}

	@Override
	public void shutdown() {
		executorService.shutdown();
	}

	@Override
	public List<Runnable> shutdownNow() {
		executorService.shutdownNow();
		return null;
	}

	@Override
	public <T> Future<T> submit(Callable<T> task) {
		return executorService.submit(new AASCallable<T>(task));
	}

	@Override
	public Future<?> submit(Runnable task) {
		return executorService.submit(new AASRunnable(task));
	}

	@Override
	public <T> Future<T> submit(Runnable task, T result) {
		return executorService.submit(new AASRunnable(task), result);
	}

	@Override
	public void execute(Runnable command) {
		executorService.execute(new AASRunnable(command));
	}

	class AASRunnable implements Runnable {
		private Runnable runnable = null;
		String securityContext = null;

		public AASRunnable(Runnable runnable) {
			this.runnable = runnable;

			// runs in main thread
			this.securityContext = callerSecurityContext.get();
		}

		public void run() {
			// runs in another thread
			callerSecurityContext.set(securityContext);
			runnable.run();
		}
	}

	class AASCallable<V> implements Callable<V> {
		private Callable<V> callable = null;
		String securityContext = null;

		public AASCallable(Callable<V> callable) {
			this.callable = callable;

			// runs in main thread
			this.securityContext = callerSecurityContext.get();
		}

		public V call() throws Exception {
			// runs in another thread
			callerSecurityContext.set(securityContext);

			return callable.call();
		}
	}

	public ThreadLocal<String> getCallerSecurityContext() {
		return callerSecurityContext;
	}

	public void setCallerSecurityContext(
			ThreadLocal<String> callerSecurityContext) {
		this.callerSecurityContext = callerSecurityContext;
	}

	public ScheduledExecutorService getExecutorService() {
		return executorService;
	}

	public void setExecutorService(ScheduledExecutorService executorService) {
		this.executorService = executorService;
	}
}