package gr.uoa.di.driver.enabling.issn;

import eu.dnetlib.api.enabling.ISSNService;
import eu.dnetlib.api.enabling.ISSNServiceException;
import eu.dnetlib.domain.enabling.Notification;
import eu.dnetlib.domain.enabling.Subscription;
import gr.uoa.di.driver.util.ServiceLocator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;

public class SNManagerImpl implements SNManager {
	private Logger logger = Logger.getLogger(this.getClass());
	
	private ScheduledExecutorService executor = null;

	private ServiceLocator<ISSNService> snLocator = null;
	private int timeToLive = 3600;
	private int threadPoolSize = 5;
	
	/** topic -> suscription task map **/
	private Map<String, SubscriptionTask> tasks = new HashMap<String, SNManagerImpl.SubscriptionTask>();
	
	/** subscriptionid -> subscription task map */
	private Map<String, SubscriptionTask> idMap = new HashMap<String, SNManagerImpl.SubscriptionTask>();
	
	public void init() {
		logger.debug("Creating executor with " + this.threadPoolSize + " threads");
		this.executor = Executors.newScheduledThreadPool(this.threadPoolSize);
	}

	@Override
	public void subscribe(Subscription sub, NotificationListener listener) {
		//TODO
		//TEMPORARY FIX FOR REJECTED SUBSCRIPTIONS. ISSUE:1288
		if (!sub.getTopic().split("/")[2].equals("*")) {
			SubscriptionTask task = tasks.get(sub.getTopic());
			if (task == null) {
					logger.debug("New topic: " + sub.getTopic());
					task = new SubscriptionTask(sub, listener);
					
					tasks.put(sub.getTopic(), task);
	
					sub.setTimeToLive(this.timeToLive);
					
					executor.execute(task);
			} else {
				logger.debug("Topic already exists, adding listener");
				task.getListeners().add(listener);
			} 
		} else {
			logger.debug("subscription for resourceType: " + sub.getTopic().split("/")[1] + " ignored due to missing identifier");
		}
	}

	@Override
	public void unsubscribe(Subscription sub, NotificationListener listener) {
		SubscriptionTask task = tasks.get(sub.getTopic());
		
		if (task != null)
			task.getListeners().remove(listener);
	}

	@Override
	public void unsubscribe(NotificationListener listener) {
		for (SubscriptionTask task:tasks.values())
			task.getListeners().remove(listener);
	}

	@Override
	public void notify(Notification notification) {
		SubscriptionTask task = idMap.get(notification.getSubscriptionId());
		
		if (task == null || task.getListeners().size() == 0)
			logger.warn("No listeners for notification: " + notification.getTopic());
		else {
			for (NotificationListener listener:task.getListeners()) {
				this.executor.execute(new NotificationTask(listener, notification));
			}
		}
	}
	
	public void shutdown() {
		this.executor.shutdownNow();
		
		logger.debug("Removing active subscriptions");
		for (String id:idMap.keySet()) {
			try {
				snLocator.getService().unsubscribe(id);
			} catch (ISSNServiceException e) {
				logger.error("Error removing subscription: " + id);
			}
		}
	}
	
	public ServiceLocator<ISSNService> getSnLocator() {
		return snLocator;
	}

	public void setSnLocator(ServiceLocator<ISSNService> snLocator) {
		this.snLocator = snLocator;
	}

	public int getTimeToLive() {
		return timeToLive;
	}

	public void setTimeToLive(int timeToLive) {
		this.timeToLive = timeToLive;
	}

	public int getThreadPoolSize() {
		return threadPoolSize;
	}

	public void setThreadPoolSize(int threadPoolSize) {
		this.threadPoolSize = threadPoolSize;
	}

	private class NotificationTask implements Runnable {
		private NotificationListener listener = null;
		private Notification notification = null;
		
		public NotificationTask(NotificationListener listener, Notification notification) {
			this.listener = listener;
			this.notification = notification;
		}
		
		@Override
		public void run() {
			try {
				this.listener.processNotification(notification);
			} catch (Exception e) {
				logger.error("Error calling listener", e);
			}
		}
	}
	
	private class SubscriptionTask implements Runnable {
		private Subscription subscription = null;
		private List<NotificationListener> listeners = new ArrayList<NotificationListener>();
		
		public SubscriptionTask(Subscription sub, NotificationListener listener) {
			this.subscription = sub;
			this.listeners.add(listener);
		}
		
		@Override
		public void run() {
			try {
				if (this.subscription.getId() == null) {
					logger.debug("New subscription. Subscribing and scheduling for refresh");
					
					try {
						subscription.setId(snLocator.getService().subscribe(
								subscription.getEpr(), subscription.getTopic(), 
								subscription.getTimeToLive()));
					} catch (ISSNServiceException e) {
						logger.error("Error adding topic: " + subscription.getTopic(), e);
					}
					
					executor.schedule(this, (long) (this.subscription.getTimeToLive() * .9), TimeUnit.SECONDS);
					
					idMap.put(subscription.getId(), this);
				} else {
					if (this.listeners.size() > 0) {
						logger.debug("Subscription " + subscription.getId() + " is already subscribed. Refreshing and rescheduling");
						
						try {
							snLocator.getService().renew(subscription.getId(), subscription.getTimeToLive());
						} catch (ISSNServiceException e) {
							logger.error("Error refreshing subscription " + subscription.getId(), e);
						}
						
						executor.schedule(this, (long) (this.subscription.getTimeToLive() * .9), TimeUnit.SECONDS);
					} else {
						logger.debug("No listeners for this subscription. Removing it");
						
						try {
							snLocator.getService().unsubscribe(this.subscription.getId());
						} catch (ISSNServiceException e) {
							logger.error("Error removing subscription " + subscription.getId(), e);
						}
						
						idMap.remove(subscription.getId());
						tasks.remove(this.subscription);
					}
				}
			} catch (Exception e) {
				logger.error("Error processing subscription");
			}
		}
		
		public List<NotificationListener> getListeners() {
			return this.listeners;
		}
	}
}