package eu.dnetlib.msro.workflows.nodes.validation;

import org.apache.commons.lang.StringUtils;

import com.googlecode.sarasvati.Engine;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.msro.workflows.nodes.BlackboardJobNode;
import eu.dnetlib.msro.workflows.nodes.blackboard.BlackboardWorkflowJobListener;

/**
 * This node send the BB message to the ValidatorService to validate records from an mdstore.
 * <p>
 * See ticket https://issue.openaire.research-infrastructures.eu/issues/538 for specs.
 * </p>
 * 
 * @author andrea.mannocci, alessia
 * 
 */
public class ValidatorServiceBlackboardJobNode extends BlackboardJobNode {

	public static final String VALIDATE_ACTION = "VALIDATE";
	public static final String VALIDATION_TYPE = "DNET";

	public static final String VALIDATOR_EPR_PARAM_NAME = "outputResultSetEpr";

	private String xQueryForValidatorServiceId;
	private String outputEprName = VALIDATOR_EPR_PARAM_NAME;

	// VALIDATOR JOB PARAM VALUES
	private String mdstoreId;
	private String guidelines;
	private String datasourceID;
	private String datasourceNamespacePrefix;
	private String datasourceName;
	private String datasourceBaseUrl;
	private String recordsSampleSize;
	/** Json key-value map of other parameters that will be stored in the validator back-end as they are. **/
	private String extraParams;

	/**
	 * <p>
	 * Parameters to set for the VALIDATE BB job:
	 * 
	 * <pre>
	 * <PARAMETER name="mdstoreId" value=" "/>
	 * <PARAMETER name="guidelines" value="openaire3.0"/> //values: openaire3.0, openaire2.0, openaire2.0_data, driver
	 * <PARAMETER name="type" value="DNET"/>
	 * <PARAMETER name="datasourceID" value=""/>
	 * <PARAMETER name="datasourceNamespacePrefix" value=""/> //OPTIONAL
	 * <PARAMETER name="datasourceName" value=""/> //OPTIONAL
	 * <PARAMETER name="submissionDate" value="YYYY-MM-DD HH24:MI:SS"/>
	 * <PARAMETER name="baseUrl" value=""/> //OPTIONAL
	 * <PARAMETER name="records" value="#records"/> //OPTIONAL
	 * <PARAMETER name="extraParams" value="json key-value map"/> //OPTIONAL
	 * </pre>
	 * 
	 * </p>
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.msro.workflows.nodes.BlackboardJobNode#prepareJob(eu.dnetlib.enabling.tools.blackboard.BlackboardJob,
	 *      com.googlecode.sarasvati.NodeToken)
	 */
	@Override
	protected void prepareJob(final BlackboardJob job, final NodeToken token) throws Exception {
		// required params
		job.setAction(VALIDATE_ACTION);
		job.getParameters().put("mdstoreId", mdstoreId);
		job.getParameters().put("guidelines", guidelines);
		job.getParameters().put("type", VALIDATION_TYPE);
		job.getParameters().put("datasourceID", datasourceID);
		job.getParameters().put("submissionDate", DateUtils.now_ISO8601());

		// optional params
		if (StringUtils.isNotBlank(datasourceNamespacePrefix)) {
			job.getParameters().put("datasourceNamespacePrefix", datasourceNamespacePrefix);
		}
		if (StringUtils.isNotBlank(datasourceName)) {
			job.getParameters().put("datasourceName", datasourceName);
		}
		if (StringUtils.isNotBlank(datasourceBaseUrl)) {
			job.getParameters().put("baseUrl", datasourceBaseUrl);
		}
		if (StringUtils.isNotBlank(getRecordsSampleSize())) {
			job.getParameters().put("records", recordsSampleSize);
		}
		if (StringUtils.isNotBlank(extraParams)) {
			job.getParameters().put("extraParams", extraParams);
		}

	}

	/**
	 * <p>
	 * Parameters in the VALIDATE BB job response:
	 * 
	 * <pre>
	 * <PARAMETER name="error" value=""/>
	 * <PARAMETER name="outputResultSetEpr" value=""/>
	 * <PARAMETER name="jobId" value=""/>
	 * </pre>
	 * 
	 * </p>
	 * 
	 * 
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.msro.workflows.nodes.BlackboardJobNode#generateBlackboardListener(com.googlecode.sarasvati.Engine,
	 *      com.googlecode.sarasvati.NodeToken)
	 */
	@Override
	protected BlackboardWorkflowJobListener generateBlackboardListener(final Engine engine, final NodeToken token) {
		return new BlackboardWorkflowJobListener(engine, token) {

			@Override
			protected void onDone(final BlackboardJob job) {
				token.getEnv().setAttribute(outputEprName, job.getParameters().get(VALIDATOR_EPR_PARAM_NAME));
				token.getEnv().setAttribute("validatorJobId", job.getParameters().get("jobId"));
				String error = job.getParameters().get("error");
				if (StringUtils.isNotBlank(error)) {
					token.getEnv().setAttribute("validatorError", error);
				}
				super.onDone(job);
			}
		};
	}

	@Override
	protected String obtainServiceId(final NodeToken token) {
		try {
			return getServiceLocator().getService(ISLookUpService.class).getResourceProfileByQuery(xQueryForValidatorServiceId);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	public String getGuidelines() {
		return guidelines;
	}

	public void setGuidelines(final String guidelines) {
		this.guidelines = guidelines;
	}

	public String getRecordsSampleSize() {
		return recordsSampleSize;
	}

	public void setRecordsSampleSize(final String recordsSampleSize) {
		this.recordsSampleSize = recordsSampleSize;
	}

	public String getMdstoreId() {
		return mdstoreId;
	}

	public void setMdstoreId(final String mdstoreId) {
		this.mdstoreId = mdstoreId;
	}

	public String getDatasourceID() {
		return datasourceID;
	}

	public void setDatasourceID(final String datasourceID) {
		this.datasourceID = datasourceID;
	}

	public String getDatasourceNamespacePrefix() {
		return datasourceNamespacePrefix;
	}

	public void setDatasourceNamespacePrefix(final String datasourceNamespacePrefix) {
		this.datasourceNamespacePrefix = datasourceNamespacePrefix;
	}

	public String getDatasourceName() {
		return datasourceName;
	}

	public void setDatasourceName(final String datasourceName) {
		this.datasourceName = datasourceName;
	}

	public String getDatasourceBaseUrl() {
		return datasourceBaseUrl;
	}

	public void setDatasourceBaseUrl(final String datasourceBaseUrl) {
		this.datasourceBaseUrl = datasourceBaseUrl;
	}

	public String getExtra_param() {
		return extraParams;
	}

	public void setExtra_param(final String extra_param) {
		this.extraParams = extra_param;
	}

	public String getxQueryForValidatorServiceId() {
		return xQueryForValidatorServiceId;
	}

	public void setxQueryForValidatorServiceId(final String xQueryForValidatorServiceId) {
		this.xQueryForValidatorServiceId = xQueryForValidatorServiceId;
	}

}
