package eu.dnetlib.msro.openaireplus.workflows.nodes;

import java.text.SimpleDateFormat;
import java.util.Date;

import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;
import com.mongodb.DBObject;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Sorts;
import eu.dnetlib.common.logging.DnetLogger;
import eu.dnetlib.common.logging.dao.DnetLoggerMongoDao;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

public class PrepareLodParamJobNode extends SimpleJobNode {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(PrepareLodParamJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM
	private static final String SYSTEM_PROFILE_ID = "system:profileId";
	private static final String LOG_DATE = "log:date";
	private static final String IS_COMPLETED_SUCCESSFULLY = "system:isCompletedSuccessfully";

	@Value("${dnet.openaire.lod.jsonrels}")
	private String lodJsonRels;

	@Value("${dnet.openaire.lod.wfId}")
	private String lodWfId;

	@Value("${dnet.openaire.lod.default.date}")
	private String defaultLodDate;

	@Autowired
	private DnetLogger dnetLogger;

	@Autowired
	private DnetLoggerMongoDao mongoDao;

	private final SimpleDateFormat dateFormat = new SimpleDateFormat("dd-MM-yyyy");

	@Override
	protected String execute(final NodeToken token) throws Exception {

		log.info("prepare LOD update job");

		token.getEnv().setAttribute("lod_jsonRels", lodJsonRels);
		token.getEnv().setAttribute("lod_lastExecutionDate", getFormattedDate());

		return Arc.DEFAULT_ARC;
	}

	/**
	 * Queries the MSRO logger to find the last time the LOD update workflow was successfully executed.
	 * @return
	 *  the formatted date
	 */
	private String getFormattedDate() {
		final DBObject dbo = getLoggerDao()
				.find(Filters.and(Filters.eq(SYSTEM_PROFILE_ID, lodWfId), Filters.eq(IS_COMPLETED_SUCCESSFULLY, "true")))
				.sort(Sorts.descending(LOG_DATE))
				.limit(1)
				.first();

		if (dbo == null || !dbo.containsField(LOG_DATE)) {
			log.warn(String.format("Cannot find log entry for LOD update workflow, using default date: '%s'", defaultLodDate));
			return defaultLodDate;
		}

		final String lastExecutionDate = dateFormat.format(new Date((Long) dbo.get(LOG_DATE)));
		log.debug(String.format("Last LOD execution date: '%s'", lastExecutionDate));
		return lastExecutionDate;
	}

	private MongoCollection<DBObject> getLoggerDao() {
		return mongoDao.getDb().getCollection(dnetLogger.getName(), DBObject.class);
	}

}
