package eu.dnetlib.msro.openaireplus.api;

import java.io.IOException;
import java.util.List;
import javax.annotation.Resource;

import com.google.common.collect.Lists;
import com.google.gson.Gson;
import eu.dnetlib.common.rmi.DNetRestDocumentation;
import eu.dnetlib.data.index.CloudIndexClient;
import eu.dnetlib.data.index.CloudIndexClientFactory;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.msro.openaireplus.api.objects.ResultEntry;
import eu.dnetlib.msro.openaireplus.utils.OafToIndexRecordFactory;
import eu.dnetlib.msro.rmi.MSROException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.app.VelocityEngine;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

/**
 * Created by michele on 11/11/15.
 */
@Controller
@DNetRestDocumentation
public class OpenaireResultSubmitter {

	private static final Log log = LogFactory.getLog(OpenaireResultSubmitter.class);

	public class IndexDsInfo {

		private final String indexBaseUrl;
		private final String indexDsId;
		private final String format;
		private final String coll;

		public IndexDsInfo(final String indexBaseUrl, final String indexDsId, final String format, final String coll) {
			this.indexBaseUrl = indexBaseUrl;
			this.indexDsId = indexDsId;
			this.format = format;
			this.coll = coll;
		}

		public String getIndexBaseUrl() {
			return indexBaseUrl;
		}

		public String getIndexDsId() {
			return indexDsId;
		}

		public String getFormat() {
			return format;
		}

		public String getColl() {
			return coll;
		}

	}

	@Value(value = "oaf.schema.location")
	private String oafSchemaLocation;

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Resource
	private OafToIndexRecordFactory oafToIndexRecordFactory;

	@Resource
	private RecentResultsQueue recentResultsQueue;

	@Resource(name = "openaireplusApisVelocityEngine")
	private VelocityEngine velocityEngine;

	@Value(value = "${openaireplus.msro.api.findSolrIndexUrl.xquery}")
	private ClassPathResource findSolrIndexUrl;

	@Value(value = "${openaireplus.msro.api.findIndexDsInfo.xquery}")
	private ClassPathResource findIndexDsInfo;

	@Deprecated
	@RequestMapping(value = {"/api/publications/feedJson", "/api/results/feedJson"}, method = RequestMethod.POST)
	public
	@ResponseBody
	boolean feedObjectJson(@RequestParam(value = "json", required = true) final String json,
			@RequestParam(value = "commit", required = false, defaultValue = "true") final boolean commit) throws MSROException {
		final ResultEntry pub = new Gson().fromJson(json, ResultEntry.class);
		return feedObject(pub, commit);
	}


	@RequestMapping(value = {"/api/results/feedObject"} , method = RequestMethod.POST)
	public
	@ResponseBody
	String feedResult(@RequestBody final ResultEntry pub,
			@RequestParam(value = "commit", required = false, defaultValue = "true") final boolean commit)
			throws MSROException {
		try {
			if(feedObject(pub, commit)) {
				return ResultEntry.calculateObjId(pub.getOriginalId(), pub.getCollectedFromId(), serviceLocator.getService(ISLookUpService.class));
			}
			else return null;
		} catch (final Throwable e) {
			throw new MSROException("Error adding publication: " + e.getMessage(), e);
		}
	}

	@RequestMapping(value = {"/api/publications/feedObject"} , method = RequestMethod.POST)
	public
	@ResponseBody
	boolean feedObject(@RequestBody final ResultEntry pub,
			@RequestParam(value = "commit", required = false, defaultValue = "true") final boolean commit)
			throws MSROException {

		checkParameters(pub);
		final List<IndexDsInfo> idxList;
		try {
			idxList = calculateCurrentIndexDsInfo();
			if (idxList == null || idxList.isEmpty()) {
				throw new MSROException("Cannot add result: " + pub.getOriginalId() + " : No public Search Service found");
			}
			if (idxList.size() > 1) log.warn("Found more than 1 public search service");
			final String oafRecord = pub.asOafRecord(velocityEngine, serviceLocator.getService(ISLookUpService.class), oafSchemaLocation);

			for (IndexDsInfo idx : idxList) {
				CloudIndexClient idxClient = null;
				try {
					idxClient = CloudIndexClientFactory.newIndexClient(idx.getIndexBaseUrl(), idx.getColl(), false);
					idxClient.feed(oafRecord, idx.getIndexDsId(), oafToIndexRecordFactory.newTransformer(idx.getFormat()), commit);
				} catch (final Throwable e) {
					throw new MSROException("Error adding publication: " + e.getMessage(), e);
				} finally {
					if (idxClient != null) {
						idxClient.close();
					}
				}
			}
			recentResultsQueue.add(oafRecord);
			return true;
		} catch (final Throwable e) {
			throw new MSROException("Error adding publication: " + e.getMessage(), e);
		}

	}



	@RequestMapping(value = "/api/results", method = RequestMethod.DELETE)
	public
	@ResponseBody
	boolean deleteResult(
			@RequestParam(value = "originalId", required = true) final String originalId,
			@RequestParam(value = "collectedFromId", required = true) final String collectedFromId,
			@RequestParam(value = "commit", required = false, defaultValue = "true") final boolean commit) throws MSROException {

		return deleteResultPost(originalId, collectedFromId, commit);
	}

	@RequestMapping(value = {"/api/publications/deleteObject", "/api/results/delete"}, method = RequestMethod.POST)
	public
	@ResponseBody
	boolean deleteResultPost(
			@RequestParam(value = "originalId", required = true) final String originalId,
			@RequestParam(value = "collectedFromId", required = true) final String collectedFromId,
			@RequestParam(value = "commit", required = false, defaultValue = "true") final boolean commit) throws MSROException {

		final List<IndexDsInfo> idxList;
		try {
			idxList = calculateCurrentIndexDsInfo();

			if (idxList == null || idxList.isEmpty()) {
				throw new MSROException("Cannot delete result: " + originalId + " : No public Search Service found");
			}
			if (idxList.size() > 1) log.warn("Found more than 1 public search service");

			final String objId = ResultEntry.calculateObjId(originalId, collectedFromId, serviceLocator.getService(ISLookUpService.class));

			for (IndexDsInfo idx : idxList) {
				CloudIndexClient idxClient = null;
				try {
					idxClient = CloudIndexClientFactory.newIndexClient(idx.getIndexBaseUrl(), idx.getColl(), false);
					idxClient.remove(objId, commit);
					log.info("Deleted result with id: " + objId + " from: " + idx.getIndexBaseUrl());
				} catch (final Throwable e) {
					throw new MSROException("Error deleting publication: " + e.getMessage(), e);
				} finally {
					if (idxClient != null) {
						idxClient.close();
					}
				}
			}
			recentResultsQueue.remove(objId);
			return true;
		} catch (IOException | ISLookUpException e) {
			throw new MSROException("Error deleting publication: " + e.getMessage(), e);
		}
	}

	@ExceptionHandler(Exception.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	public
	@ResponseBody
	ErrorMessage handleException(final Exception e) {
		log.error("Error in direct index API", e);
		return new ErrorMessage(e);
	}

	private void checkParameters(ResultEntry pub) throws MSROException{
		if (StringUtils.isBlank(pub.getOriginalId())) { throw new MSROException("A required field is missing: originalId"); }
		if (StringUtils.isBlank(pub.getTitle())) { throw new MSROException("A required field is missing: title"); }
		if (StringUtils.isBlank(pub.getUrl())) { throw new MSROException("A required field is missing: url"); }
		if (StringUtils.isBlank(pub.getLicenseCode())) { throw new MSROException("A required field is missing: licenseCode"); }
		if (StringUtils.isBlank(pub.getResourceType())) { throw new MSROException("A required field is missing: resourceType"); }
		if (StringUtils.isBlank(pub.getCollectedFromId())) { throw new MSROException("A required field is missing: collectedFromId"); }
		if (StringUtils.isBlank(pub.getType())) { throw new MSROException("A required field is missing: type"); }
	}

	private List<IndexDsInfo> calculateCurrentIndexDsInfo() throws IOException, ISLookUpException {
		List<IndexDsInfo> list = Lists.newArrayList();

		final String queryUrl = IOUtils.toString(findSolrIndexUrl.getInputStream());
		final String queryDs = IOUtils.toString(findIndexDsInfo.getInputStream());

		final ISLookUpService lu = serviceLocator.getService(ISLookUpService.class);
		final String indexBaseUrl = lu.getResourceProfileByQuery(queryUrl);

		final List<String> idxDs = lu.quickSearchProfile(queryDs);
		for (String idx : idxDs) {
			final String[] arr = idx.split("@@@");
			list.add(new IndexDsInfo(indexBaseUrl, arr[0].trim(), arr[1].trim(), arr[2].trim()));
		}
		return list;
	}

	public class ErrorMessage {

		private final String message;
		private final String stacktrace;

		public ErrorMessage(final Exception e) {
			this(e.getMessage(), ExceptionUtils.getStackTrace(e));
		}

		public ErrorMessage(final String message, final String stacktrace) {
			this.message = message;
			this.stacktrace = stacktrace;
		}

		public String getMessage() {
			return this.message;
		}

		public String getStacktrace() {
			return this.stacktrace;
		}

	}

}
