package eu.dnetlib.msro.openaireplus.workflows.nodes.contexts;

import java.io.IOException;
import java.util.*;

import com.google.common.collect.Maps;
import com.ximpleware.AutoPilot;
import com.ximpleware.VTDException;
import com.ximpleware.VTDGen;
import com.ximpleware.VTDNav;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.joda.time.DateTime;

public class BuildH2020FETTaxonomy {

	private static final Log log = LogFactory.getLog(BuildH2020FETTaxonomy.class);

	private Iterator<String> projects;
	private Map<String, Map<String, Map<String, List<Info>>>> dic;

	public BuildH2020FETTaxonomy() {
		dic = Maps.newHashMap();
	}

	public void setIterator(Iterator<String> projects) {
		this.projects = projects;
	}

	public String parseProjects() throws Exception {
		while (projects.hasNext())
			parse(projects.next());
		return getTaxonomy();
	}

	public String getTaxonomy() throws Exception {
		Document taxonomy = DocumentHelper.createDocument();
		Element root1 = taxonomy.addElement("RESOURCE_PROFILE");
		createHeader(root1);
		Element body = root1.addElement("BODY");
		Element configurationElement = body.addElement("CONFIGURATION");
		createContext(configurationElement);
		body.addElement("STATUS");
		body.addElement("SECURITY_PARAMETERS");
		return taxonomy.getDocument().asXML();

	}

	private void createContext(final Element configurationElement) {
		Element context = configurationElement.addElement("context");
		context.addAttribute("type", "community");
		context.addAttribute("id", "fet-h2020");
		context.addAttribute("label", "FET");
		for (Map.Entry<String, Map<String, Map<String, List<Info>>>> entry : dic.entrySet()) {
			String type = entry.getKey();
			Element cat = context.addElement("category");
			String catid = "fet-h2020::" + type.toLowerCase();
			cat.addAttribute("id", catid);
			cat.addAttribute("label", "FET " + type);
			cat.addAttribute("claim", "false");
			int count = 0;
			for (Map.Entry<String, Map<String, List<Info>>> entryst : dic.get(type).entrySet()) {
				String subtype = entryst.getKey();
				for (Map.Entry<String, List<Info>> entryp : dic.get(type).get(subtype).entrySet()) {
					String opt = entryp.getKey();
					count++;
					Element subcat = cat.addElement("concept");
					String subcatid = catid + "::" + count;
					subcat.addAttribute("id", subcatid);
					Element pa = subcat.addElement("param");
					pa.addAttribute("name", "CALL_ID");
					List<Info> projectRows = dic.get(type).get(subtype).get(opt);
					if (!projectRows.isEmpty()) {
						for (Info row : projectRows) {
							createProjectConcept(row, subcat, subcatid);
						}
						subcat.addAttribute("label", projectRows.get(0).getCallname());
						subcat.addAttribute("claim", "false");
						pa.setText(projectRows.get(0).getCallID());
					} else throw new RuntimeException("The lis of projects for " + subcatid + " is empty. This exception should never be thrown.");
				}
			}
		}
	}

	private void createProjectConcept(Info row, Element father, String prefix) {
		String code = row.getCode();
		String acronym = row.getAcronym();
		String title = row.getTitle();
		String projcallid = row.getProjcallid();
		Element concept = father.addElement("concept");
		concept.addAttribute("id", prefix + "::" + code);
		concept.addAttribute("label", title);
		concept.addAttribute("claim", "true");
		Element p = concept.addElement("param");
		p.addAttribute("name", "CD_PROJ_ID");
		p.setText(code);
		p = concept.addElement("param");
		p.addAttribute("name", "CD_CALL_ID");
		p.setText(projcallid);
		p = concept.addElement("param");
		p.addAttribute("name", "CD_PROJECT_NUMBER");
		p.setText(code);
		p = concept.addElement("param");
		p.addAttribute("name", "CD_ACRONYM");
		p.setText(acronym);
		p = concept.addElement("param");
		p.addAttribute("name", "CD_FRAMEWORK");
		p.setText("H2020");
	}

	private void createHeader(final Element profile) {
		Element header = profile.addElement("HEADER");
		Element rs = header.addElement("RESOURCE_IDENTIFIER");
		rs.addAttribute("value", "");
		Element rt = header.addElement("RESOURCE_TYPE");
		rt.addAttribute("value", "ContextDSResourceType");
		Element rk = header.addElement("RESOURCE_KIND");
		rk.addAttribute("value", "ContextDSResources");
		Element ru = header.addElement("RESOURCE_URI");
		ru.addAttribute("value", "");
		Element daoc = header.addElement("DATE_OF_CREATION");
		daoc.addAttribute("value", DateTime.now().toString());
	}

	private boolean parse(String project) throws VTDException, IOException {
		boolean fet = false;
		String callid = "";
		String projectid = "";
		VTDGen vg = new VTDGen();
		vg.setDoc(project.getBytes("UTF-8"));
		vg.parse(false);
		VTDNav vn = vg.getNav();
		AutoPilot ap = new AutoPilot(vn);
		ap.selectXPath("//metadata/ROWS/ROW[@table=\"projects\"]");

		while (ap.evalXPath() != -1) {
			AutoPilot ap1 = new AutoPilot(vn);
			ap1.selectXPath("./FIELD[@name=\"call_identifier\"]");
			if (ap1.evalXPath() != -1)
				callid = vn.toNormalizedString(vn.getText());
			ap1.clearVariableExprs();
			vn.toElement(VTDNav.PARENT);
			ap1.selectXPath("./FIELD[@name=\"optional1\"]");
			if (ap1.evalXPath() != -1)
				projectid = vn.toNormalizedString(vn.getText());
			if (callid.contains("FET")) {
				String type;
				if (callid.contains("OPEN"))
					type = "OPEN";
				else if (callid.contains("PROACT"))
					type = "PROACT";
				else if (callid.contains("HPC"))
					type = "HPC";
				else
					type = "FLAG";
				vn.toElement(VTDNav.ROOT);
				fet = true;
				insert(type, callid, projectid, new Info(vn));
			}
		}
		return fet;
	}

	private void insert(String type, String callid, String projcallid, Info row) {
		Map<String, Map<String, List<Info>>> dopen;
		Map<String, List<Info>> entry;
		List<Info> projects;
		if (dic.containsKey(type)) {
			dopen = dic.get(type);
			if (dopen.containsKey(callid)) {
				entry = dopen.get(callid);
				if (entry.containsKey(projcallid)) {
					entry.get(projcallid).add(row);
				} else {
					projects = new ArrayList<>();
					projects.add(row);
					entry.put(projcallid, projects);
				}
			} else {
				projects = new ArrayList<>();
				entry = new HashMap<>();
				projects.add(row);
				entry.put(projcallid, projects);
				dopen.put(callid, entry);
			}
		} else {
			projects = new ArrayList<>();
			entry = new HashMap<>();
			dopen = new HashMap<>();
			projects.add(row);
			entry.put(projcallid, projects);
			dopen.put(callid, entry);
			dic.put(type, dopen);
		}
	}

	public void parseFETProject(int projects_number) throws VTDException, IOException{
		int parsed = 0;
		while (projects.hasNext() && parsed < projects_number) {
			if (parse(projects.next()))
				parsed++;
		}
		log.debug(parsed);
	}

	private class Info {

		VTDNav vn;
		private String code, acronym, title, call_identifier, projcallid, callname;

		public String getCode() {
			return code;
		}

		public String getAcronym() {
			return acronym;
		}

		public String getTitle() {
			return title;
		}

		public String getCallID() {
			return call_identifier;
		}

		public String getProjcallid() {
			return projcallid;
		}

		public String getCallname() {
			return callname;
		}

		public Info(VTDNav vn) {
			this.vn = vn;
			parse();
		}

		private void parse() {
			try {
				AutoPilot ap = new AutoPilot(vn);
				ap.selectXPath("//FIELD");
				while (ap.evalXPath() != -1) {
					String aname = vn.toNormalizedString(vn.getAttrVal("name"));
					String text = "";
					if (vn.getText() != -1)
						text = vn.toNormalizedString(vn.getText());
					if (aname.equals("code"))
						code = text;
					else if (aname.equals("acronym"))
						acronym = text;
					else if (aname.equals("title"))
						title = text;
					else if (aname.equals("call_identifier"))
						call_identifier = text;
					else if (aname.equals("optional1"))
						projcallid = text;
					else if (aname.equals("optional2"))
						callname = text;
				}
			} catch (Exception e) {
			}
		}

	}

}
