package eu.dnetlib.msro.openaireplus.api;

import javax.annotation.Resource;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.velocity.app.VelocityEngine;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.google.gson.Gson;

import eu.dnetlib.data.index.CloudIndexClient;
import eu.dnetlib.data.index.CloudIndexClientFactory;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.msro.openaireplus.api.objects.PublicationEntry;
import eu.dnetlib.msro.openaireplus.utils.OafToIndexRecordFactory;
import eu.dnetlib.msro.rmi.MSROException;

/**
 * Created by michele on 11/11/15.
 */
@Controller
public class SinglePublicationSubmitter {

	public class IndexDsInfo {

		private final String indexBaseUrl;
		private final String indexDsId;
		private final String format;
		private final String coll;

		public IndexDsInfo(final String indexBaseUrl, final String indexDsId, final String format, final String coll) {
			this.indexBaseUrl = indexBaseUrl;
			this.indexDsId = indexDsId;
			this.format = format;
			this.coll = coll;
		}

		public String getIndexBaseUrl() {
			return indexBaseUrl;
		}

		public String getIndexDsId() {
			return indexDsId;
		}

		public String getFormat() {
			return format;
		}

		public String getColl() {
			return coll;
		}

	}

	@Value(value = "oaf.schema.location")
	private String oafSchemaLocation;

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Resource
	private OafToIndexRecordFactory oafToIndexRecordFactory;

	@Resource
	private RecentPublicationsQueue recentPublicationsQueue;

	@Resource(name = "openaireplusApisVelocityEngine")
	private VelocityEngine velocityEngine;

	@Value(value = "${openaireplus.msro.api.findSolrIndexUrl.xquery}")
	private ClassPathResource findSolrIndexUrl;

	@Value(value = "${openaireplus.msro.api.findIndexDsInfo.xquery}")
	private ClassPathResource findIndexDsInfo;

	@RequestMapping(value = "/api/publications/feedJson", method = RequestMethod.POST)
	public @ResponseBody boolean addPublication(@RequestParam(value = "json", required = true) final String json) throws MSROException {
		final PublicationEntry pub = new Gson().fromJson(json, PublicationEntry.class);
		return addPublication(pub);
	}

	@RequestMapping(value = "/api/publications/feedObject", method = RequestMethod.POST)
	public @ResponseBody boolean addPublication(@RequestBody final PublicationEntry pub) throws MSROException {

		if (StringUtils.isBlank(pub.getOriginalId())) { throw new MSROException("A required field is missing: originalId"); }
		if (StringUtils.isBlank(pub.getTitle())) { throw new MSROException("A required field is missing: title"); }
		if (StringUtils.isBlank(pub.getUrl())) { throw new MSROException("A required field is missing: url"); }
		if (StringUtils.isBlank(pub.getLicenseCode())) { throw new MSROException("A required field is missing: licenceCode"); }
		if (StringUtils.isBlank(pub.getResourceType())) { throw new MSROException("A required field is missing: resourceType"); }
		if (StringUtils.isBlank(pub.getCollectedFromId())) { throw new MSROException("A required field is missing: collectedFromId"); }
		if (StringUtils.isBlank(pub.getType())) { throw new MSROException("A required field is missing: type"); }

		/*
		 * if (StringUtils.isBlank(pub.getHostedById())) { throw new MSROException("A required field is missing: hostedById"); }
		 */

		CloudIndexClient idxClient = null;

		try {
			final IndexDsInfo idx = calculateCurrentIndexDsInfo();

			idxClient = CloudIndexClientFactory.newIndexClient(idx.getIndexBaseUrl(), idx.getColl(), false);

			final String oafRecord = pub.asOafRecord(velocityEngine, serviceLocator.getService(ISLookUpService.class), oafSchemaLocation);

			recentPublicationsQueue.add(oafRecord);

			return (idxClient.feed(oafRecord, idx.getIndexDsId(), oafToIndexRecordFactory.newTransformer(idx.getFormat())) == 0);
		} catch (final Throwable e) {
			throw new MSROException("Error adding publication: " + e.getMessage(), e);
		} finally {
			if (idxClient != null) {
				idxClient.close();
			}
		}
	}

	@RequestMapping(value = "/api/publications/deleteObject", method = RequestMethod.POST)
	public @ResponseBody boolean deletePublication(
			@RequestParam(value = "originalId", required = true) final String originalId,
			@RequestParam(value = "collectedFromId", required = true) final String collectedFromId) throws MSROException {

		final CloudIndexClient idxClient = null;
		try {
			final IndexDsInfo idx = calculateCurrentIndexDsInfo();

			final String objId = PublicationEntry.calculateObjId(originalId, collectedFromId, serviceLocator.getService(ISLookUpService.class));

			CloudIndexClientFactory.newIndexClient(idx.getIndexBaseUrl(), idx.getColl(), false).remove(objId);

			recentPublicationsQueue.remove(objId);

			return true;
		} catch (final Throwable e) {
			throw new MSROException("Error adding publication: " + e.getMessage(), e);
		} finally {
			if (idxClient != null) {
				idxClient.close();
			}
		}
	}

	private IndexDsInfo calculateCurrentIndexDsInfo() throws Exception {
		final String queryUrl = IOUtils.toString(findSolrIndexUrl.getInputStream());
		final String queryDs = IOUtils.toString(findIndexDsInfo.getInputStream());

		final ISLookUpService lu = serviceLocator.getService(ISLookUpService.class);
		final String indexBaseUrl = lu.getResourceProfileByQuery(queryUrl);
		final String[] arr = lu.getResourceProfileByQuery(queryDs).split("@@@");

		return new IndexDsInfo(indexBaseUrl, arr[0].trim(), arr[1].trim(), arr[2].trim());
	}

}
