package eu.dnetlib.msro.openaireplus.workflows.nodes.index;

import java.io.StringReader;
import java.util.List;
import javax.annotation.Resource;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;
import eu.dnetlib.data.index.CloudIndexClient;
import eu.dnetlib.data.index.CloudIndexClientException;
import eu.dnetlib.data.index.CloudIndexClientFactory;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.miscutils.functional.UnaryFunction;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;
import eu.dnetlib.msro.openaireplus.api.RecentPublicationsQueue;
import eu.dnetlib.msro.openaireplus.utils.OafToIndexRecordFactory;
import eu.dnetlib.msro.rmi.MSROException;
import eu.dnetlib.msro.workflows.nodes.AsyncJobNode;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.common.SolrInputDocument;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;

/**
 * Created by michele on 15/12/15.
 */
public class FeedMissingClaimsJobNode extends AsyncJobNode {

	private static final Log log = LogFactory.getLog(FeedMissingClaimsJobNode.class);
	private RecentPublicationsQueue queue;
	private OafToIndexRecordFactory oafToIndexRecordFactory;

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Value(value = "${openaireplus.msro.api.findSolrIndexUrl.xquery}")
	private ClassPathResource findSolrIndexUrl;

	@Override
	protected String execute(final NodeToken nodeToken) throws Exception {

		final String format =
				nodeToken.getEnv().hasAttribute("format") ? nodeToken.getEnv().getAttribute("format") : nodeToken.getFullEnv().getAttribute("format");
		final String coll = format + "-index-openaire";
		final String indexDsId = nodeToken.getEnv().getAttribute("index_id");
		final String baseUrl = calculateIndexBaseUrl();

		CloudIndexClient idxClient = null;

		final ApplyXslt xslt = oafToIndexRecordFactory.newTransformer(format);

		try {
			idxClient = CloudIndexClientFactory.newIndexClient(baseUrl, coll, false);

			final List<SolrInputDocument> records = Lists.newArrayList(
					Iterables.transform(Iterables.filter(queue, new IsNotIndexed(idxClient)), new CreateSolrDocument(idxClient, indexDsId, xslt)));

			idxClient.feed(records, null);

		} catch (Throwable e) {
			log.error("Error feeding missing claims", e);
			throw new MSROException("Error feeding missing claims: " + e.getMessage(), e);
		} finally {
			if (idxClient != null) {
				idxClient.close();
			}
		}

		return Arc.DEFAULT_ARC;
	}

	public RecentPublicationsQueue getQueue() {
		return queue;
	}

	@Required
	public void setQueue(final RecentPublicationsQueue queue) {
		this.queue = queue;
	}

	public OafToIndexRecordFactory getOafToIndexRecordFactory() {
		return oafToIndexRecordFactory;
	}

	@Required
	public void setOafToIndexRecordFactory(final OafToIndexRecordFactory oafToIndexRecordFactory) {
		this.oafToIndexRecordFactory = oafToIndexRecordFactory;
	}

	private class IsNotIndexed implements Predicate<String> {

		private final CloudIndexClient idxClient;

		private final SAXReader reader = new SAXReader();

		public IsNotIndexed(final CloudIndexClient idxClient) {
			this.idxClient = idxClient;
		}

		@Override
		public boolean apply(final String s) {
			try {
				final String id = reader.read(new StringReader(s)).valueOf("//*[local-name() = 'objIdentifier']");
				return !idxClient.isRecordIndexed(id);
			} catch (Throwable e) {
				log.error("Error searching record: " + s, e);
				throw new RuntimeException(e);
			}
		}

	}

	private class CreateSolrDocument implements Function<String, SolrInputDocument> {

		private final CloudIndexClient idxClient;
		private final String indexDsId;
		private final UnaryFunction<String, String> toIndexRecord;

		public CreateSolrDocument(final CloudIndexClient idxClient, final String indexDsId, final UnaryFunction<String, String> toIndexRecord) {
			this.idxClient = idxClient;
			this.indexDsId = indexDsId;
			this.toIndexRecord = toIndexRecord;
		}

		@Override
		public SolrInputDocument apply(final String s) {
			try {
				return idxClient.prepareSolrDocument(s, indexDsId, toIndexRecord);
			} catch (CloudIndexClientException e) {
				throw new RuntimeException(e);
			}
		}

	}

	private String calculateIndexBaseUrl() throws Exception {
		final String query = IOUtils.toString(findSolrIndexUrl.getInputStream());
		return serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(query);
	}
}
