package eu.dnetlib.msro.openaireplus.workflows.nodes.hbase;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Iterables;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.data.hadoop.rmi.HadoopBlackboardActions;
import eu.dnetlib.data.hadoop.rmi.HadoopJobType;
import eu.dnetlib.data.hadoop.rmi.HadoopService;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.msro.workflows.nodes.BlackboardJobNode;

public class SubmitHadoopJobNode extends BlackboardJobNode {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(SubmitHadoopJobNode.class);

	@Resource
	private UniqueServiceLocator serviceLocator;

	private String hadoopJob;

	private String cluster;

	private boolean simulation = false;

	@Override
	protected String obtainServiceId(final NodeToken token) {
		return getServiceLocator().getServiceId(HadoopService.class);
	}

	@Override
	protected void prepareJob(final BlackboardJob job, final NodeToken token) throws Exception {
		String type = getJobType(getHadoopJob());

		log.info("submitting job " + getHadoopJob() + " type: " + type);

		job.setAction(type);
		job.getParameters().put("job.name", getHadoopJob());
		job.getParameters().put("cluster", cluster(token));
		job.getParameters().put("simulation", String.valueOf(isSimulation()));

		job.getParameters().putAll(parseJsonParameters(token));
	}

	private String cluster(final NodeToken token) {
		if (token.getEnv().hasAttribute("cluster")) {
			String cluster = token.getEnv().getAttribute("cluster");
			log.info("found override value in wfEnv for 'cluster' param: " + cluster);
			return cluster;
		}
		return getCluster();
	}

	/**
	 * reads the job type for the given job name
	 *
	 * @param jobName
	 * @return
	 * @throws ISLookUpException
	 */
	private String getJobType(final String jobName) throws ISLookUpException {
		List<String> res =
				serviceLocator.getService(ISLookUpService.class).quickSearchProfile(
						"/RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'HadoopJobConfigurationDSResourceType']//HADOOP_JOB[./@name='" + jobName
						+ "']/@type/string()");
		if (res.isEmpty()) { throw new IllegalStateException("unable to find job type for job: " + jobName); }

		final HadoopJobType type = HadoopJobType.valueOf(Iterables.getOnlyElement(res));

		switch (type) {
		case mapreduce:
			return HadoopBlackboardActions.SUBMIT_MAPREDUCE_JOB.toString();
		case admin:
			return HadoopBlackboardActions.SUBMIT_ADMIN_JOB.toString();
		case oozie:
			return HadoopBlackboardActions.SUBMIT_OOZIE_JOB.toString();
		default:
			throw new IllegalStateException("undefined job type: " + type.toString());
		}
	}

	public String getHadoopJob() {
		return hadoopJob;
	}

	public void setHadoopJob(final String hadoopJob) {
		this.hadoopJob = hadoopJob;
	}

	public String getCluster() {
		return cluster;
	}

	public void setCluster(final String cluster) {
		this.cluster = cluster;
	}

	public boolean isSimulation() {
		return simulation;
	}

	public void setSimulation(final boolean simulation) {
		this.simulation = simulation;
	}
}
