package eu.dnetlib.msro.openaireplus.workflows.nodes.dedup;

import java.io.StringReader;
import java.util.Iterator;
import java.util.List;
import java.util.Queue;

import javax.annotation.Resource;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.collect.Lists;
import com.google.common.collect.Queues;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.enabling.resultset.IterableResultSetFactory;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.functionality.modular.ui.dedup.SimilarityMeshBuilder;
import eu.dnetlib.miscutils.collections.Pair;
import eu.dnetlib.msro.workflows.nodes.AsyncJobNode;

public class BuildSimilarityMeshJobNode extends AsyncJobNode {

	private static final Log log = LogFactory.getLog(BuildSimilarityMeshJobNode.class);

	/** The result set factory. */
	@Resource(name = "iterableResultSetFactory")
	private IterableResultSetFactory resultSetFactory;

	/** The result set client factory. */
	@Resource(name = "resultSetClientFactory")
	private ResultSetClientFactory resultSetClientFactory;

	private StringTemplate similarity;

	private String inputEprParam;

	private String outputEprParam;

	@Override
	protected String execute(NodeToken token) throws Exception {

		final String inputEpr = token.getEnv().getAttribute(getInputEprParam());

		final Iterator<String> rsClient = resultSetClientFactory.getClient(inputEpr).iterator();
		final Queue<Object> queue = Queues.newLinkedBlockingQueue();
		final SAXReader reader = new SAXReader();

		if (rsClient.hasNext()) {
			populateQueue(queue, reader, rsClient.next());
		}

		final W3CEndpointReference eprOut = resultSetFactory.createIterableResultSet(new Iterable<String>() {

			@Override
			public Iterator<String> iterator() {
				return new Iterator<String>() {

					@Override
					public boolean hasNext() {
						synchronized (queue) {
							return !queue.isEmpty();
						}
					}

					@Override
					@SuppressWarnings("unchecked")
					public String next() {
						synchronized (queue) {
							final Object o = queue.poll();
							while (queue.isEmpty() && rsClient.hasNext()) {
								populateQueue(queue, reader, rsClient.next());
							}
							return buildSimilarity((Pair<String, String>) o);
						}
					}

					@Override
					public void remove() {
						throw new UnsupportedOperationException();
					}
				};
			}
		});

		token.getEnv().setAttribute(getOutputEprParam(), eprOut.toString());

		return Arc.DEFAULT_ARC;
	}

	private void populateQueue(final Queue<Object> q, final SAXReader r, final String xml) {
		try {
			final Document d = r.read(new StringReader(xml));
			final String groupid = d.valueOf("//FIELD[@name='id']");
			final List<?> items = d.selectNodes("//FIELD[@name='group']/ITEM");
			final String entitytype = d.valueOf("//FIELD[@name='entitytype']");
			final List<String> group = Lists.newArrayList();
			for (final Object id : items) {
				group.add(((Node) id).getText());
			}
			// compute the full mesh
			final String type = String.valueOf(Type.valueOf(entitytype).getNumber());
			final List<Pair<String, String>> mesh = SimilarityMeshBuilder.build(type, group);
			// total += mesh.size();
			if (log.isDebugEnabled()) {
				log.debug(String.format("built mesh for group '%s', size %d", groupid, mesh.size()));
			}
			for (final Pair<String, String> p : mesh) {
				if (log.isDebugEnabled()) {
					log.debug(String.format("adding to queue: %s", p.toString()));
				}
				q.add(p);
			}
		} catch (final DocumentException e) {
			log.error("invalid document: " + xml);
		}
	}

	private String buildSimilarity(final Pair<String, String> p) {
		final StringTemplate template = new StringTemplate(getSimilarity().getTemplate());

		template.setAttribute("source", p.getKey());
		template.setAttribute("target", p.getValue());

		final String res = template.toString();
		return res;
	}

	public String getInputEprParam() {
		return inputEprParam;
	}

	public void setInputEprParam(String inputEprParam) {
		this.inputEprParam = inputEprParam;
	}

	public String getOutputEprParam() {
		return outputEprParam;
	}

	public void setOutputEprParam(String outputEprParam) {
		this.outputEprParam = outputEprParam;
	}

	public StringTemplate getSimilarity() {
		return similarity;
	}

	@Required
	public void setSimilarity(StringTemplate similarity) {
		this.similarity = similarity;
	}

}
