CREATE TABLE IF NOT EXISTS dli_object (
  dli_id            VARCHAR(1000) PRIMARY KEY,
  pid               VARCHAR(2000),
  pid_type          VARCHAR(50) DEFAULT 'doi',
  title             TEXT [],
  authors           TEXT [],
  creation_date     VARCHAR(50),
  type              VARCHAR(250),
  completion_status VARCHAR(50)
);

CREATE TABLE IF NOT EXISTS pid_tmp (
  pid          VARCHAR(1000),
  ds_id        VARCHAR(250),
  publisher_id VARCHAR(250)
);

CREATE TABLE IF NOT EXISTS dli_object_tmp (
  dli_identifier    VARCHAR(1000),
  pid               VARCHAR(2000),
  pid_type          VARCHAR(50) DEFAULT 'doi',
  title             TEXT [],
  authors           TEXT [],
  creation_date     VARCHAR(50),
  type              VARCHAR(250),
  completion_status VARCHAR(50)
);


CREATE TABLE IF NOT EXISTS dli_datasource (
  ds_id      VARCHAR(250) PRIMARY KEY,
  name       VARCHAR(250) NOT NULL,
  web        VARCHAR(250),
  exportable BOOLEAN DEFAULT TRUE,
  typology   VARCHAR(250),
  icon_uri   VARCHAR(250),
  latitude   DOUBLE PRECISION,
  longitude  DOUBLE PRECISION
);


CREATE TABLE IF NOT EXISTS dli_relation (
  id_relation                    TEXT PRIMARY KEY,
  source_record                  VARCHAR(1000) NOT NULL REFERENCES dli_object (dli_id) ON DELETE CASCADE,
  source_pid                     VARCHAR(2000) NOT NULL,
  target_type                    VARCHAR(250),
  target_pid                     VARCHAR(2000) NOT NULL,
  target_pid_type                VARCHAR(250)  NOT NULL,
  target_title                   TEXT,
  target_authors                 TEXT [],
  relation_semantic              VARCHAR(100),
  relationship_completion_status VARCHAR(50)
);

create index on dli_relation(target_pid);

CREATE TABLE IF NOT EXISTS dli_relation_tmp (
  id_relation                    TEXT,
  source_record                  VARCHAR(1000),
  source_pid                     VARCHAR(2000),
  target_type                    VARCHAR(250),
  target_pid                     VARCHAR(2000),
  target_pid_type                VARCHAR(250),
  target_title                   TEXT,
  target_authors                 TEXT [],
  relation_semantic              VARCHAR(100),
  relationship_completion_status VARCHAR(50)
);

CREATE TABLE IF NOT EXISTS record_provenance (
  ds_id             VARCHAR(250) NOT NULL REFERENCES dli_datasource (ds_id) ON DELETE CASCADE,
  dli_id            VARCHAR(250) NOT NULL REFERENCES dli_object (dli_id) ON DELETE CASCADE,
  provision_mode    VARCHAR(50)  NOT NULL,
  completion_status VARCHAR(50)  NOT NULL,
  contribution      TEXT         NOT NULL,
  collection_date   DATE DEFAULT CURRENT_DATE,
  publisher         VARCHAR(250),
  PRIMARY KEY (ds_id, dli_id)
);

CREATE TABLE IF NOT EXISTS relation_provenance (
  ds_id             VARCHAR(250) NOT NULL REFERENCES dli_datasource (ds_id) ON DELETE CASCADE,
  id_relation       TEXT         NOT NULL REFERENCES dli_relation (id_relation) ON DELETE CASCADE,
  provision_mode    VARCHAR(50)  NOT NULL,
  completion_status VARCHAR(50)  NOT NULL,
  collection_date   DATE DEFAULT CURRENT_DATE,
  PRIMARY KEY (ds_id, id_relation)
);

CREATE TABLE IF NOT EXISTS record_provenance_tmp (
  ds_id             VARCHAR(250),
  dli_id            VARCHAR(250),
  provision_mode    VARCHAR(50),
  contribution      TEXT,
  completion_status VARCHAR(50),
  publisher         VARCHAR(250)
);

CREATE TABLE IF NOT EXISTS relation_provenance_tmp (
  ds_id             VARCHAR(250),
  id_relation       TEXT,
  provision_mode    VARCHAR(50),
  completion_status VARCHAR(50)
);




CREATE MATERIALIZED VIEW new_dli_records AS
  SELECT
    dli_identifier                AS dli_id,
    min(pid)                      AS pid,
    min(lower(pid_type))          AS pid_type,
    min(creation_date)            AS creation_date,
    min(lower(type))              AS type,
    min(lower(completion_status)) AS status,
    max(title)                    AS title,
    max(authors)                  AS authors
  FROM dli_object_tmp
  GROUP BY dli_identifier;

CREATE MATERIALIZED VIEW new_dli_record_to_add AS SELECT dli_id
                                                  FROM new_dli_records EXCEPT SELECT dli_id
                                                                              FROM dli_object;


CREATE MATERIALIZED VIEW new_dli_relation AS
  SELECT
    id_relation                         AS id_relation,
    max(source_record)                  AS source_record,
    max(source_pid)                     AS source_pid,
    max(target_pid)                     AS target_pid,
    max(lower(target_pid_type))         AS target_pid_type,
    max(target_title)                   AS target_title,
    min(relationship_completion_status) AS relationship_completion_status,
    max(target_type)                    AS target_type,
    max(target_authors)                 AS target_authors,
    max(relation_semantic)              AS relation_semantic
  FROM dli_relation_tmp
  GROUP BY id_relation;

CREATE MATERIALIZED VIEW new_dli_relation_to_add AS SELECT id_relation
                                                    FROM new_dli_relation EXCEPT SELECT id_relation
                                                                                 FROM dli_relation;

CREATE MATERIALIZED VIEW new_record_provenance AS
  SELECT
    min(ds_id)             AS ds_id,
    max(publisher)         AS publisher,
    dli_id                 AS dli_id,
    min(provision_mode)    AS provision_mode,
    min(completion_status) AS completion_status,
    min(contribution)      AS contribution
  FROM record_provenance_tmp
  GROUP BY dli_id;

CREATE MATERIALIZED VIEW new_record_provenance_to_add AS SELECT
                                                           ds_id,
                                                           dli_id
                                                         FROM new_record_provenance EXCEPT SELECT
                                                                                             ds_id,
                                                                                             dli_id
                                                                                           FROM record_provenance;

CREATE MATERIALIZED VIEW new_relation_provenance AS
  SELECT
    min(ds_id)             AS ds_id,
    id_relation            AS id_relation,
    min(provision_mode)    AS provision_mode,
    min(completion_status) AS completion_status
  FROM relation_provenance_tmp
  GROUP BY id_relation;

CREATE MATERIALIZED VIEW new_relation_provenance_to_add AS SELECT
                                                             ds_id,
                                                             id_relation
                                                           FROM new_relation_provenance EXCEPT SELECT
                                                                                                 ds_id,
                                                                                                 id_relation
                                                                                               FROM relation_provenance;

CREATE MATERIALIZED VIEW object_exported AS
  SELECT
    o.dli_id,
    o.pid,
    o.pid_type,
    o.title,
    o.authors,
    o.creation_date,
    o.type,
    o.completion_status,
    array_to_json(array_agg(rel.*)) AS provenance
  FROM dli_object o LEFT JOIN record_provenance rel ON (o.dli_id = rel.dli_id)
  GROUP BY o.dli_id;

CREATE MATERIALIZED VIEW relation_exported AS
  SELECT
    r.source_record                  AS record_id,
    r.source_pid                     AS source_pid,
    r.target_type                    AS target_type,
    r.target_pid                     AS target_pid,
    r.target_pid_type                AS target_pid_type,
    r.target_title                   AS title,
    r.target_authors                 AS authors,
    r.relation_semantic              AS relation,
    r.relationship_completion_status AS completion_status,
    array_to_json(array_agg(rel.*))  AS datasources
  FROM dli_relation r LEFT JOIN relation_provenance rel ON (rel.id_relation = r.id_relation)
  GROUP BY r.id_relation;

CREATE MATERIALIZED VIEW exported_view AS
  SELECT
    o.dli_id                               AS dli_id,
    min(o.pid)                             AS pid,
    min(pid_type)                          AS pid_type,
    max(o.title)                           AS title,
    max(o.authors)                         AS authors,
    max(o.creation_date)                   AS creation_date,
    max(o.type)                            AS type,
    max(o.completion_status)               AS completion_status,
    array_to_json(array_agg(o.provenance)) AS provenance,
    array_to_json(array_agg(rel.*))        AS relations
  FROM object_exported o LEFT OUTER JOIN relation_exported rel ON (o.dli_id = rel.record_id)
  GROUP BY o.dli_id;

CREATE MATERIALIZED VIEW new_dli_object_to_upsert AS
  SELECT n.*
  FROM new_dli_records n LEFT JOIN dli_object o ON (n.dli_id = o.dli_id)
  WHERE n.status = 'complete' AND o.completion_status = 'incomplete';

CREATE MATERIALIZED VIEW new_dli_relation_to_upsert AS
  SELECT n.*
  FROM new_dli_relation n LEFT JOIN dli_relation r ON (n.id_relation = r.id_relation)
  WHERE n.relationship_completion_status = 'complete' AND r.relationship_completion_status = 'incomplete';


INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::r3d100010327', 'RCSB', 'http://www.rcsb.org/pdb/home/home.do', TRUE,
   'http://mgl.scripps.edu/people/goodsell/illustration/pdb/PDB-logo-small.jpg', 40.523677, -74.469108,
   'dataPublisher');

INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::r3d100010197', 'Cambridge Crystallographic Data Centre', 'http://www.ccdc.cam.ac.uk/pages/Home.aspx', TRUE,
   'http://iycr2014.org/__data/assets/image/0019/82180/CCDC_logo.JPG', 52.19763, 0.1267, 'dataPublisher');

INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::r3d100010134', 'PANGAEA', 'http://pangaea.de', TRUE, 'http://pangaea.de/PICS/pangaea_logo_150x107.png',
   53.16781, 8.64938,
   'dataPublisher');

INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::r3d100010216', '3TU Datacentrum', 'http://datacentrum.3tu.nl/en/home', TRUE,
   'https://pbs.twimg.com/profile_images/507441262274674688/m8GuM3M7_400x400.png', 52.0027092, 4.3731207,
   'dataPublisher');

INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::r3d100010578', 'IEDA', 'http://www.iedadata.org', TRUE,
   'http://www.iedadata.org/sites/www.iedadata.org/files/arthemia_logo.jpg', 41.0032789, -73.9110483, 'dataPublisher');

INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::system', 'System Deduction', 'http://dliservice.research-infrastructures.eu/', TRUE,
   'http://dliservice.research-infrastructures.eu/images/dli_logo_big.png', 43.7185687, 10.4202594, 'resolver');

INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::thomsonreuters', 'Thomson Reuters', 'http://thomsonreuters.com', TRUE,
   'http://agency.reuters.com/content/dam/openweb/images/brand-assets/tr-logos/tr_ahz_rgb_pos.png/_jcr_content/renditions/original',
   40.7564945, -73.9891776, 'publicationPublisher');

INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::ieee', 'IEEE', 'https://www.ieee.org/ INDEX.html', TRUE,
   'https://www.ieee.org/ucm/groups/ PUBLIC /@ieee/@web/@org/@globals/documents/images/ieee_logo_mb_tagline.gif',
   40.6954239, -74.039781, 'publicationPublisher');

INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::crossref', 'CrossRef', 'http://www.crossref.org/', TRUE,
   'http://www.crossref.org/imagesHeader/cr_logotype.gif',
   51.749677, -1.270762, 'publicationCatalogue');

INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::r3d100010255', 'ICPSR', 'https://www.icpsr.umich.edu', TRUE,
   'http://www.icpsr.umich.edu/icpsrweb/AHRQMCC/images/icpsr-logo.png', 37.71859, -95.712891, 'dataPublisher');

INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::chorus', 'Chorus', 'http://www.chorusaccess.org', FALSE,
   'http://02c494f.netsolhost.com/_attachments/images/chorus%20logo1-web.jpg', 40.607428, -74.1536757,
   'publicationIntersection');


INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::dataciteResolver', 'DataCite Resolver', 'http://datacite.org', TRUE,
   'http://datacite.org/sites/ DEFAULT /files/DataCite_header_final1_1.png', 52.38142, 9.72007, 'resolver');


INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::datacite', 'Datasets IN DataCite', 'http://datacite.org', TRUE,
   'http://datacite.org/sites/ DEFAULT /files/DataCite_header_final1_1.png', 52.38142, 9.72007, 'dataCatalogue');


INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::elsevier', 'Elsevier', 'http://www.eu.elsevierhealth.com', TRUE,
   'http://secure-ecsd.elsevier.com/covers/app/ELSHSBS/ukne/img/new_logo.png', 40.7420392, -73.9874114,
   'publicationPublisher');


INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::nature', 'Springer Nature', 'http://www.nature.com/', TRUE,
   'http://www.nature.com/npg_/images/section-images/company- INFO -big.jpg', 51.53434, -0.12139,
   'publicationPublisher');


INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::r3d100010464', 'Australian NATIONAL DATA Service', 'http://www.ands.org.au/', TRUE,
   'http://www.ands.org.au/assets/images/ands- FULL -b.png', -35.2854146, 149.1155786, 'dataCatalogue');


INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::openaireResolver', 'OpenAIRE Resolver', 'https://www.openaire.eu/', TRUE,
   'https://www.openaire.eu/images/site_images/OpenAIREplus_logo.png', 50.84348, 4.38256, 'resolver');


INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::openaire', 'OpenAIRE', 'https://www.openaire.eu/', TRUE,
   'https://www.openaire.eu/images/site_images/OpenAIREplus_logo.png', 50.84348, 4.38256, 'publicationCatalogue');


INSERT INTO dli_datasource (ds_id, name, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::pbmedResolver', 'PubMed Resolver', 'http://www.ncbi.nlm.nih.gov/pubmed', TRUE,
   'https://upload.wikimedia.org/wikipedia/commons/thumb/f/fb/US-NLM-PubMed-Logo.svg/720px-US-NLM-PubMed-Logo.svg.png',
   53.480759, -2.242631, 'resolver');

INSERT INTO dli_datasource (ds_id, NAME, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::pubmed', 'PubMed', 'http://www.ncbi.nlm.nih.gov/pubmed', TRUE,
   'https://upload.wikimedia.org/wikipedia/commons/thumb/f/fb/US-NLM-PubMed-Logo.svg/720px-US-NLM-PubMed-Logo.svg.png',
   53.480759, -2.242631, 'publicationPublisher');


INSERT INTO dli_datasource (ds_id, NAME, web, exportable, icon_uri, latitude, longitude, typology) VALUES
  ('dli::mnd', 'Mendeley Data and published articles', 'https://www.mendeley.com', TRUE,
   'http://4.bp.blogspot.com/-f5BI0-PogJg/VFawjZsHzTI/AAAAAAAAAqY/mkJmPciqljo/s1600/mendeley.png',
   51.5218346, -0.0884449, 'publicationPublisher');


