/**
 *
 */
package eu.dnetlib.msro.workflows.dli.resolver;

import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Lists;

import eu.dnetlib.msro.workflows.dli.manager.DLIDBManager;
import eu.dnetlib.msro.workflows.dli.model.DLICompletionStatus;
import eu.dnetlib.msro.workflows.dli.model.DLIObject;



/**
 * The Class WorkerResolver.
 *
 * @author sandro
 */
public class WorkerResolver implements Callable<Boolean> {


	/** The end queue. */
	public static DLIObject END_QUEUE = new DLIObject();


	/** The Constant log. */
	private static final Log log = LogFactory.getLog(WorkerResolver.class); // NOPMD by marko on 11/24/08 5:02 PM


	/** The doi resolvers. */
	private final List<DOIResolver> doiResolvers;


	/** The queue. */
	private final BlockingQueue<DLIObject> pidQueue;


	/** The thread count. */
	private final int threadCount;

	private final static String[] resolvablePidType=new String[]{"doi", "openaire", "pdb", "openaire", "pubmedid", "pmcid", "url"};

	/** The manager. */
	private final DLIDBManager manager;




	/**
	 * Instantiates a new worker resolver.
	 *
	 * @param doiResolvers the doi resolvers
	 * @param pidQueue the pid queue
	 * @param threadCount the thread count
	 * @param manager the manager
	 */
	public WorkerResolver(final List<DOIResolver> doiResolvers, final BlockingQueue<DLIObject> pidQueue, final int threadCount, final DLIDBManager manager) {
		this.pidQueue= pidQueue;
		this.doiResolvers = doiResolvers;
		this.threadCount = threadCount;
		this.manager = manager;
	}





	/**
	 * {@inheritDoc}
	 * @see java.util.concurrent.Callable#call()
	 */
	@Override
	public Boolean call() throws Exception {
		log.info(threadCount+" is Started!");

		DLIObject nextRecord = null;
		try {
			nextRecord = this.pidQueue.take();
			if (nextRecord == END_QUEUE) {
				log.debug(this.threadCount + ":Found terminator record");
				this.pidQueue.put(END_QUEUE);
				return true;
			}
		} catch (InterruptedException e) {
			log.error(this.threadCount + ":Error on taking an element on queue", e);
		}
		List<DLIObject> resolvedbuffer = Lists.newArrayList();
		List<DLIObject> unresolvedbuffer = Lists.newArrayList();
		while ((nextRecord != null) && (nextRecord != END_QUEUE)) {
			try {
				if (resolvePid(nextRecord)) {
					resolvedbuffer.add(nextRecord);
				} else {
					unresolvedbuffer.add(nextRecord);
				}

				if (resolvedbuffer.size()>100) {
					manager.updateResolved(resolvedbuffer);
					resolvedbuffer.clear();
				}
				if (unresolvedbuffer.size()>100) {
					manager.updateUnresolved(unresolvedbuffer);
					unresolvedbuffer.clear();
				}
			} catch (Exception e) {
				log.error("failed to resolve record with pid: "+nextRecord.getPid());
			}
			nextRecord = this.pidQueue.take();
			log.debug(this.threadCount + ":Taken item");
			if (nextRecord == END_QUEUE) {
				manager.updateResolved(resolvedbuffer);
				manager.updateUnresolved(unresolvedbuffer);
				log.debug(this.threadCount + ":Found terminator record");
				this.pidQueue.put(END_QUEUE);
			}

		}
		return true;
	}

	private boolean isPidTypeResolvable(final String pidType) {
		for(String s: resolvablePidType) {
			if (s.equals(pidType))
				return true;
		}
		return false;
	}





	/**
	 * Resolve pid.
	 *
	 * @param nextRecord the next record
	 * @return true, if successful
	 */
	private boolean resolvePid(final DLIObject nextRecord) {
		String pid = nextRecord.getPid();
		String pidType = nextRecord.getPidType();
		if(pidType!= null) {
			pidType = pidType.toLowerCase().trim();
			if (isPidTypeResolvable(pidType.toLowerCase().trim())) {
				for (DOIResolver resolverItem: doiResolvers) {
					DLIObject data = resolverItem.retrieveDOI(pid, nextRecord.getPidType());
					if (data != null) {
						nextRecord.setCompletionStatus(DLICompletionStatus.complete.toString());
						nextRecord.setDatasourceProvenance(data.getDatasourceProvenance());
						nextRecord.setTitles(data.getTitles());
						nextRecord.setType(data.getType());
						nextRecord.setAuthors(data.getAuthors());
						nextRecord.setDate(data.getDate());
						return true;
					}
				}
			}
			return false;
		} else {
			nextRecord.setCompletionStatus(DLICompletionStatus.failed_to_resolve.toString());
			return false;
		}
	}
}
