/**
 *
 */
package eu.dnetlib.msro.workflows.dli.resolver;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Lists;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import eu.dnetlib.msro.workflows.dli.model.DLIObject;
import eu.dnetlib.msro.workflows.dli.model.DLIObjectType;
import eu.dnetlib.msro.workflows.dli.model.DLIProvenance;

/**
 * @author sandro
 *
 */
public class PubMedResolver implements DOIResolver {

	private static final Log log = LogFactory.getLog(PubMedResolver.class); // NOPMD by marko on 11/24/08 5:02 PM

	private static final String query = "http://www.ebi.ac.uk/europepmc/webservices/rest/search/query=%s&format=json";

	private static final String PUBMED_NS_PREFIX = "dli::pbmedResolver";

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.msro.workflows.dli.resolver.DOIResolver#retrieveDOI(java.lang.String)
	 */
	@Override
	public DLIObject retrieveDOI(final String doi, final String doiType) {
		if (doiType == null)
			return null;
		String requestUrl = null;
		String type = null;
		if (doiType.toLowerCase().trim().equals("pubmedid")) {
			requestUrl = String.format(query, "ext_id:" + doi);
			type = "pubmedid";
		} else if (doiType.toLowerCase().trim().equals("pmcid")) {
			requestUrl = String.format(query, "pmcid:" + doi);
			type = "pmcid";
		}

		if (requestUrl == null)
			return null;

		String result = requestDOI(requestUrl);

		return parsingResponse(result, type);

	}

	private DLIObject parsingResponse(final String response, final String type) {
		if (response == null) return null;
		JsonElement jElement = new JsonParser().parse(response);
		JsonObject jobject = jElement.getAsJsonObject();
		if (jobject.has("hitCount")) {

			long total = jobject.get("hitCount").getAsLong();

			if (total == 0) return null;
			DLIObject responseObj = new DLIObject();

			JsonElement hits = ((JsonObject) jobject.get("resultList")).get("result");

			JsonArray hitsObject = hits.getAsJsonArray();

			for (JsonElement elem : hitsObject) {
				String idType = "id";
				if (type.equals("pmcid")) {
					idType = type;
				} else {
					idType = "pmid";
				}

				String doi = ((JsonObject) elem).get(idType).getAsString();
				responseObj.setPid(doi);
				responseObj.setPidType(type);
				responseObj.setType(DLIObjectType.publication);
				List<String> authors = Lists.newArrayList();
				List<String> titles = Lists.newArrayList();

				// ADDING TITLES
				JsonElement titleElement = ((JsonObject) elem).get("title");
				if (titleElement != null) {
					String titleString = titleElement.getAsString();
					titles.add(titleString);
				}
				responseObj.setTitles(titles.toArray(new String[0]));
				// ADDING AUTHORS
				JsonElement contributorElement = ((JsonObject) elem).get("authorString");
				if (contributorElement != null) {
					String contributorString = contributorElement.getAsString();
					if (contributorString != null) {
						String[] splittedAuthors = contributorString.split(",");

						for (int i = 0; i < splittedAuthors.length; i++) {
							authors.add(splittedAuthors[i]);
						}
					}
				}
				responseObj.setAuthors(authors.toArray(new String[0]));

				JsonElement date = ((JsonObject) elem).get("pubYear");

				if (date.isJsonNull() == false) {
					responseObj.setDate(date.getAsString());
				}
				DLIProvenance provenance = new DLIProvenance(PUBMED_NS_PREFIX, "resolved", "complete", null, null, true);
				responseObj.fixContribution(provenance);
				responseObj.setDatasourceProvenance(Lists.newArrayList(provenance));
				return responseObj;

			}

			return null;
		}
		return null;

	}

	/**
	 * Request doi.
	 *
	 * @param doi
	 *            the doi
	 * @return the string
	 */
	private String requestDOI(final String URL) {

		try {
			final URL myURl = new URL(URL);
			final String response = IOUtils.toString(myURl.openStream());
			return response;
		} catch (MalformedURLException e) {
			log.error("Error on request DOI, request :" + URL, e);
			return null;
		} catch (IOException e) {
			log.error("Error on request DOI, request :" + URL, e);
			return null;
		}

	}

}
