package eu.dnetlib.msro.workflows.dli.resolver;

import com.google.common.collect.Lists;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;
import eu.dnetlib.msro.dli.workflows.nodes.feeding.utility.DLIRecordParser;
import eu.dnetlib.msro.workflows.dli.model.DLIObject;
import eu.dnetlib.msro.workflows.dli.model.DLIProvenance;
import net.sf.ehcache.Cache;
import net.sf.ehcache.Element;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.core.io.ClassPathResource;

import javax.annotation.Resource;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

public class OpenaireResolver implements DOIResolver {

    private final static String baseURL = "http://solr.openaire.eu:8983/solr/DMF-index-openaire_shard1_replica1/select?q=objidentifier:\"%s\"&wt=json";

    private static final Log log = LogFactory.getLog(OpenaireResolver.class); // NOPMD by marko on 11/24/08 5:02 PM

    /**
     * The cache.
     */
    @Resource(name = "openaireDOICache")
    private Cache cache;

    private ApplyXslt xsltMapper;

    private DLIRecordParser parser;

    public OpenaireResolver() {
        xsltMapper = new ApplyXslt(new ClassPathResource("eu/dnetlib/msro/dli_openaire/workflows/resolver/xslt/openaire2Dli.xslt"));
        parser = new DLIRecordParser();
    }

    @Override
    public DLIObject retrieveDOI(final String doi, final String doiType) {
        final String currentDOI = StringUtils.substringAfter(doi, "oai:dnet:");
        final Element elem = cache.get(currentDOI);
        if (elem != null) {
            log.debug("found DOI in cache");
            if (elem.getObjectValue() == null) {
                log.debug("found DOI in cache but is null");
            }
            return (DLIObject) elem.getObjectValue();

        } else {
            log.debug("Refreshing " + currentDOI + " cache...");
            DLIObject obj = parsingResponse(requestDOI(currentDOI));
            if (obj != null) {
                DLIProvenance provenance = new DLIProvenance("opnres", "resolved", "complete", null, null, true);
                obj.fixContribution(provenance);
                obj.setDatasourceProvenance(Lists.newArrayList(provenance));
                cache.put(new Element(currentDOI, obj));
            }
            return obj;
        }
    }

    /**
     * Request doi.
     *
     * @param doi the doi
     * @return the string
     */
    private String requestDOI(final String doi) {
        final String currentURL = String.format(baseURL, doi);

        try {
            final URL myURl = new URL(currentURL);
            final String response = IOUtils.toString(myURl.openStream());
            return response;
        } catch (MalformedURLException e) {
            log.error("Error on request DOI, request :" + currentURL, e);
            return null;
        } catch (IOException e) {
            log.error("Error on request DOI, request :" + currentURL, e);
            return null;
        }

    }

    private DLIObject parsingResponse(final String response) {
        if (response == null) return null;

        JsonElement jElement = new JsonParser().parse(response);
        JsonObject jobject = jElement.getAsJsonObject();
        if (jobject.has("response")) {

            long total = jobject.get("response").getAsJsonObject().get("numFound").getAsLong();

            if (total == 0) return null;

            JsonArray hits = ((JsonObject) jobject.get("response")).get("docs").getAsJsonArray();

            JsonElement s = ((JsonObject) hits.get(0)).get("__result").getAsJsonArray().get(0);
            String dlirecord = xsltMapper.evaluate(s.getAsString());

            return parser.parseRecord(dlirecord);
        }
        return null;

    }
}
