package eu.dnetlib.msro.workflows.dli.resolver;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Lists;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import eu.dnetlib.msro.workflows.dli.model.DLIObject;
import eu.dnetlib.msro.workflows.dli.model.DLIObjectType;
import eu.dnetlib.msro.workflows.dli.model.DLIProvenance;
import net.sf.ehcache.Cache;
import net.sf.ehcache.Element;

/**
 * The Class DataciteDOIResolver.
 */
public class DataciteDOIResolver implements DOIResolver {

	public static final String DATACITE_NS_PREFIX = "dli::dataciteResolver";

	/** The cache. */
	@Resource(name = "dataciteDOICache")
	private Cache cache;

	/** The Constant log. */
	private static final Log log = LogFactory.getLog(DataciteDOIResolver.class); // NOPMD by marko on 11/24/08 5:02 PM

	/** The Constant baseURL. */
	private final static String baseURL = "http://search.datacite.org/api?wt=json&fl=doi,title,contributor,creator,publisher&q=doi:";

	@Override
	public DLIObject retrieveDOI(final String doi, final String doiType) {
		final Element elem = cache.get(doi);
		if (elem != null) {
			log.debug("found DOI in cache");
			if (elem.getObjectValue() == null) {
				log.debug("found DOI in cache but is null");
			}
			return (DLIObject) elem.getObjectValue();

		} else {

			log.debug("Refreshing " + doi + " cache...");
			DLIObject obj = parsingResponse(requestDOI(doi));
			cache.put(new Element(doi, obj));
			return obj;
		}

	}

	/**
	 * Request doi.
	 *
	 * @param doi
	 *            the doi
	 * @return the string
	 */
	private String requestDOI(final String doi) {
		final String currentURL = baseURL + doi.replace(" ", "%20");

		try {
			final URL myURl = new URL(currentURL);
			final String response = IOUtils.toString(myURl.openStream());
			return response;
		} catch (MalformedURLException e) {
			log.error("Error on request DOI, request :" + currentURL, e);
			return null;
		} catch (IOException e) {
			log.error("Error on request DOI, request :" + currentURL, e);
			return null;
		}

	}

	private DLIObject parsingResponse(final String response) {
		if (response == null) return null;
		JsonElement jElement = new JsonParser().parse(response);
		JsonObject jobject = jElement.getAsJsonObject();
		if (jobject.has("response")) {

			long total = jobject.get("response").getAsJsonObject().get("numFound").getAsLong();

			if (total == 0) return null;
			DLIObject responseObj = new DLIObject();

			JsonElement hits = ((JsonObject) jobject.get("response")).get("docs");

			JsonArray hitsObject = hits.getAsJsonArray();

			for (JsonElement elem : hitsObject) {
				String doi = ((JsonObject) elem).get("doi").getAsString();
				responseObj.setPid(doi);
				responseObj.setPidType("DOI");
				responseObj.setType(DLIObjectType.dataset);
				List<String> authors = Lists.newArrayList();
				List<String> titles = Lists.newArrayList();

				// ADDING TITLES
				JsonElement titleElement = ((JsonObject) elem).get("title");
				if (titleElement != null) {
					JsonArray titleArray = titleElement.getAsJsonArray();
					for (int i = 0; i < titleArray.size(); i++) {
						titles.add(titleArray.get(i).getAsString());
					}
				}
				responseObj.setTitles(titles.toArray(new String[0]));
				// ADDING AUTHORS
				JsonElement contributorElement = ((JsonObject) elem).get("contributor");
				if (contributorElement != null) {
					JsonArray contributorArray = contributorElement.getAsJsonArray();
					for (int i = 0; i < contributorArray.size(); i++) {
						authors.add(contributorArray.get(i).getAsString());
					}
				}
				// ADDING AUTHORS
				JsonElement creatorElement = ((JsonObject) elem).get("creator");
				if (creatorElement != null) {
					JsonArray creatorArray = creatorElement.getAsJsonArray();
					for (int i = 0; i < creatorArray.size(); i++) {
						authors.add(creatorArray.get(i).getAsString());
					}
				}
				responseObj.setAuthors(authors.toArray(new String[0]));

				DLIProvenance provenance = new DLIProvenance(DATACITE_NS_PREFIX, "resolved", "complete", null, null, true);

				JsonElement publisherElement = ((JsonObject) elem).get("publisher");
				if (publisherElement != null && !publisherElement.isJsonNull()) {
					provenance.setPublisher(publisherElement.getAsString());
				}
				responseObj.fixContribution(provenance);
				responseObj.setDatasourceProvenance(Lists.newArrayList(provenance));
				return responseObj;
	 			}

			return null;
		}
		return null;

	}
}
