package eu.dnetlib.msro.workflows.dli.resolver;

import java.io.ByteArrayInputStream;
import java.util.List;
import java.util.Stack;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Lists;

import eu.dnetlib.msro.workflows.dli.model.DLIObject;
import eu.dnetlib.msro.workflows.dli.model.DLIObjectType;
import eu.dnetlib.msro.workflows.dli.model.DLIProvenance;

/**
 * The Class CrossRefParser.
 */
public class CrossRefParser {

	/** The Constant log. */
	private static final Log log = LogFactory.getLog(CrossRefParser.class); // NOPMD by marko on 11/24/08 5:02 PM

	/** The Constant CROSSREF_NS_PREFIX. */
	public static final String CROSSREF_NS_PREFIX = "dli::crossref";

	/**
	 * Parses the record.
	 *
	 * @param record
	 *            the record
	 * @return the DLI object
	 */
	public DLIObject parseRecord(final String record) {

		try {
			XMLInputFactory factory = XMLInputFactory.newInstance();
			XMLStreamReader parser = factory.createXMLStreamReader(new ByteArrayInputStream(record.getBytes()));

			DLIObject object = new DLIObject();
			Stack<String> elementStack = new Stack<String>();
			elementStack.push("/");
			List<String> titles = Lists.newArrayList();
			List<String> authors = Lists.newArrayList();
			boolean insideArticle = false;
			while (parser.hasNext()) {
				int event = parser.next();
				if (event == XMLStreamConstants.END_ELEMENT) {
					final String localName = parser.getLocalName();
					if (localName.equals("journal_article") || "series_metadata".equals(localName)) {
						log.debug("found closed tag relation");
						insideArticle = false;
					}
					elementStack.pop();
				} else if (event == XMLStreamConstants.START_ELEMENT) {
					final String localName = parser.getLocalName();
					elementStack.push(localName);
					if ("journal_article".equals(localName) || "series_metadata".equals(localName)) {
						log.debug("found open tag journal_article");
						insideArticle = true;
					} else if ("title".equals(localName) && insideArticle) {
						log.debug("found open tag title");
						parser.next();
						if (parser.hasText()) {
							titles.add(parser.getText().trim());
						}
					} else if ("surname".equals(localName) && insideArticle) {
						log.debug("found open tag fullname");
						parser.next();
						if (parser.hasText()) {
							authors.add(parser.getText().trim());
						}
					} else if ("doi".equals(localName)) {
						log.debug("found open tag relation");
						parser.next();
						if (parser.hasText()) {
							object.setPid(parser.getText().trim());
							object.setPidType("DOI");
						} else return null;

					}

				}
			}
			object.setAuthors(authors.toArray(new String[0]));
			object.setTitles(titles.toArray(new String[0]));
			DLIProvenance provenance = new DLIProvenance(CROSSREF_NS_PREFIX, "resolved", "complete", null, null, true);
			object.fixContribution(provenance);
			object.setDatasourceProvenance(Lists.newArrayList(provenance));

			object.setType(DLIObjectType.publication);
			return object;
		} catch (XMLStreamException e) {
			log.error("error during parsing record " + record, e);
			return null;
		}

	}

}
