package eu.dnetlib.msro.workflows.dli.model;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.gson.Gson;
import eu.dnetlib.miscutils.functional.xml.DnetXsltFunctions;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

// TODO: Auto-generated Javadoc

/**
 * The Class DLIObject.
 */
public class DLIObject {

	/**
	 * The doi.
	 */
	private String pid;

	/**
	 * The pid type.
	 */
	private String pidType;

	/**
	 * The titles.
	 */
	private String[] titles = new String[] {};

	/**
	 * The authors.
	 */
	private String[] authors = new String[] {};

	;

	/**
	 * The type.
	 */
	private DLIObjectType type;

	/**
	 * The datasources.
	 */
	private List<DLIProvenance> datasourceProvenance;

	/**
	 * The completion status.
	 */
	private String completionStatus;

	/**
	 * The date.
	 */
	private String date;

	/**
	 * The relations.
	 */
	private List<DLIRelation> relations;

	private HashMap<String, String> resolvedTypes = new HashMap<String, String>() {

		{
			put("pdb", "http://www.rcsb.org/pdb/explore/explore.do?structureId=%s");
			put("ncbi-n", "http://www.ncbi.nlm.nih.gov/gquery/?term=%s");
			put("pmid", "http://www.ncbi.nlm.nih.gov/pubmed/%s");
			put("pmcid", "http://www.ncbi.nlm.nih.gov/pmc/articles/%s");
			put("pubmedid", "http://www.ncbi.nlm.nih.gov/pubmed/%s");
			put("doi", "http://dx.doi.org/%s");
			put("genbank", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("nuccore", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("swiss-prot", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("arrayexpress", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("biomodels", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("bmrb", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("ena", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("genbank", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("geo", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("ensembl", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("mgi", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("bind", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("pride", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("ddbj", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("bioproject", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("embl", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
			put("sra", "http://www.ncbi.nlm.nih.gov/nucest/%s?report=genbank");
		}

	};

	/**
	 * Instantiates a new DLI object.
	 */
	public DLIObject() {

	}

	/**
	 * Instantiates a new DLI object.
	 *
	 * @param pid                  the pid
	 * @param pidType              the pid type
	 * @param titles               the titles
	 * @param authors              the authors
	 * @param type                 the type
	 * @param datasourceProvenance the datasource provenance
	 * @param relations            the relations
	 * @param status               the status
	 */
	public DLIObject(final String pid, final String pidType, final String[] titles, final String[] authors, final DLIObjectType type,
			final List<DLIProvenance> datasourceProvenance, final List<DLIRelation> relations, final String status) {
		super();
		if (pid != null) {
			this.pid = pid.replace("http://dx.doi.org/", "").replace("http://doi.org/", "").toLowerCase();
		}
		this.pidType = pidType;
		this.titles = titles;
		this.authors = authors;
		this.type = type;
		this.setDatasourceProvenance(datasourceProvenance);
		this.relations = relations;
		this.setCompletionStatus(status);
	}

	/**
	 * Gets the doi.
	 *
	 * @return the doi
	 */
	public String getPid() {
		if (pid != null)
			return pid.toLowerCase();
		return pid;
	}

	/**
	 * Sets the doi.
	 *
	 * @param pid the new pid
	 */
	public void setPid(final String pid) {
		if (pid != null) {
			this.pid = pid.replace("http://dx.doi.org/", "").replace("http://doi.org/", "");
		}
	}

	/**
	 * Check not null fields.
	 *
	 * @return the string
	 */
	private String checkNotNullFields() {

		final List<String> notNUllFields = Lists.newArrayList();
		if (!StringUtils.isEmpty(pid)) {
			notNUllFields.add("pid");
		}
		if (!StringUtils.isEmpty(pidType)) {
			notNUllFields.add("pidType");
		}
		if (!ArrayUtils.isEmpty(titles)) {
			notNUllFields.add("titles");
		}
		if (!ArrayUtils.isEmpty(authors)) {
			notNUllFields.add("authors");
		}
		if (!StringUtils.isEmpty(date)) {
			notNUllFields.add("date");
		}
		return new Gson().toJson(notNUllFields);
	}

	/**
	 * Fix contribution.
	 *
	 * @param provenance the provenance
	 */
	public void fixContribution(final DLIProvenance provenance) {
		provenance.setDatasourceContribution(checkNotNullFields());
	}

	public String getResolvedPIDUrl() {
		if (pidType == null || StringUtils.isBlank(pidType) || pid == null || StringUtils.isBlank(pid))
			return "#";

		if (resolvedTypes.containsKey(pidType)) return String.format(resolvedTypes.get(pidType), getEscapedXMLPid());

		return "#";
	}

	/**
	 * Gets the escaped xml pid.
	 *
	 * @return the escaped xml pid
	 */
	public String getEscapedXMLPid() {
		return StringEscapeUtils.escapeXml(pid);
	}

	/**
	 * Gets the titles.
	 *
	 * @return the titles
	 */
	public String[] getTitles() {
		return titles;
	}

	/**
	 * Sets the titles.
	 *
	 * @param titles the new titles
	 */
	public void setTitles(final String[] titles) {
		this.titles = titles;
	}

	/**
	 * Gets the escaped xml titles.
	 *
	 * @return the escaped xml titles
	 */
	public String[] getEscapedXMLTitles() {

		return escapedArray(titles);
	}

	/**
	 * Gets the authors.
	 *
	 * @return the authors
	 */
	public String[] getAuthors() {
		return authors;
	}

	/**
	 * Sets the authors.
	 *
	 * @param authors the new authors
	 */
	public void setAuthors(final String[] authors) {
		this.authors = authors;
	}

	/**
	 * Gets the escaped xml authors.
	 *
	 * @return the escaped xml authors
	 */
	public String[] getEscapedXMLAuthors() {
		return escapedArray(authors);
	}

	/**
	 * Gets the type.
	 *
	 * @return the type
	 */
	public DLIObjectType getType() {
		return type;
	}

	/**
	 * Sets the type.
	 *
	 * @param type the new type
	 */
	public void setType(final DLIObjectType type) {
		this.type = type;
	}

	/**
	 * Gets the escaped xml type.
	 *
	 * @return the escaped xml type
	 */
	public String getEscapedXMLType() {
		if (type != null)
			return type.toString();
		else return null;
	}

	/**
	 * Escaped array.
	 *
	 * @param inputArray the input array
	 * @return the string[]
	 */
	private String[] escapedArray(final String[] inputArray) {
		if (inputArray == null || inputArray.length == 0) return null;
		Function<String, String> function = new Function<String, String>() {

			@Override
			public String apply(final String input) {

				String bf = input;
				bf = bf.replaceAll("&", "&amp;");
				bf = bf.replaceAll("\"", "&quot;");
				bf = bf.replaceAll("'", "&apos;");
				bf = bf.replaceAll("<", "&lt;");
				bf = bf.replaceAll(">", "&gt;");

				return bf;
			}
		};

		return Lists.transform(Lists.newArrayList(inputArray), function).toArray(ArrayUtils.EMPTY_STRING_ARRAY);
	}

	/**
	 * Gets the identifier. withe the form of MD5(pid::pidType)
	 *
	 * @return the identifier
	 */
	public String getIdentifier() {
		if (StringUtils.isBlank(pid) || StringUtils.isBlank(pidType))
			throw new RuntimeException("Error pid or pidtype cannot be null");
		return DnetXsltFunctions.md5(String.format("%s::%s", pid.trim().toLowerCase(), pidType.toLowerCase().trim()));
	}

	/**
	 * Gets the relations.
	 *
	 * @return the relations
	 */
	public List<DLIRelation> getRelations() {
		return relations;
	}

	/**
	 * Sets the relations.
	 *
	 * @param relations the new relations
	 */
	public void setRelations(final List<DLIRelation> relations) {
		this.relations = relations;
	}

	/**
	 * Gets the date.
	 *
	 * @return the date
	 */
	public String getDate() {
		return date;
	}

	/**
	 * Sets the date.
	 *
	 * @param date the date to set
	 */
	public void setDate(final String date) {
		this.date = date;
	}

	/**
	 * Gets the pid type.
	 *
	 * @return the pidType
	 */
	public String getPidType() {
		return pidType;
	}

	/**
	 * Sets the pid type.
	 *
	 * @param pidType the pidType to set
	 */
	public void setPidType(final String pidType) {
		this.pidType = pidType;
	}

	/**
	 * Gets the completion status.
	 *
	 * @return the completion status
	 */
	public String getCompletionStatus() {
		return this.completionStatus;
	}

	/**
	 * Sets the completion status.
	 *
	 * @param completionStatus the new completion status
	 */
	public void setCompletionStatus(final String completionStatus) {
		this.completionStatus = completionStatus;
	}

	/**
	 * Gets the datasource provenance.
	 *
	 * @return the datasource provenance
	 */
	public List<DLIProvenance> getDatasourceProvenance() {
		return datasourceProvenance;
	}

	/**
	 * Sets the datasource provenance.
	 *
	 * @param datasourceProvenance the new datasource provenance
	 */
	public void setDatasourceProvenance(final List<DLIProvenance> datasourceProvenance) {
		this.datasourceProvenance = datasourceProvenance;
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {

		String s = "DLIObject \n\t Identifier: %s \n\t Pid:%s \n\t PIDType: %s \n\t Entity Type: %s \n\t Titles:%s \n\t Authors:%s, \n\tProvenance%s";

		return String.format(s, this.getIdentifier(), this.getPid(), this.getPidType(), this.getType(), Arrays.toString(this.getTitles()),
				Arrays.toString(this.authors), Arrays.toString(this.getDatasourceProvenance().toArray()));

	}

}
