/**
 *
 */
package eu.dnetlib.msro.workflows.dli.manager;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import eu.dnetlib.functionality.dli.datasource.DLIDatasourceInfo;
import eu.dnetlib.miscutils.collections.Pair;
import eu.dnetlib.msro.workflows.dli.model.*;
import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.dbcp.BasicDataSource;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;

import java.io.IOException;
import java.io.InputStream;
import java.sql.*;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * The Class DLIDBManager.
 *
 * @author sandro
 */
public class DLIDBManager {

    /**
     * The Constant DLI_TABLE.
     */
    private final static String DLI_TABLE = "dli_object";
    /**
     * The Constant log.
     */
    private static final Log log = LogFactory.getLog(DLIDBManager.class); // NOPMD by marko on 11/24/08 5:02 PM
    /**
     * The datasource.
     */
    @javax.annotation.Resource(name = "dliInterlinkingDataSource")
    private BasicDataSource datasource;
    /**
     * The template table.
     */
    private Resource templateTable;


    /**
     * The xml template.
     */
    private StringTemplate xmlTemplate;


    /**
     * Inits the.
     *
     * @throws SQLException the SQL exception
     * @throws IOException  Signals that an I/O exception has occurred.
     */
    public void init() throws SQLException, IOException {
        Connection conn = datasource.getConnection();
        if (existsTable(DLI_TABLE, conn)) return;
        initializeTables(conn);
    }

    /**
     * Initialize tables.
     *
     * @param connection the connection
     * @throws IOException  Signals that an I/O exception has occurred.
     * @throws SQLException the SQL exception
     */
    private void initializeTables(final Connection connection) throws IOException, SQLException {
        InputStream is = getTemplateTable().getInputStream();
        String createTableQuery = IOUtils.toString(is);
        PreparedStatement ps = connection.prepareStatement(createTableQuery);
        ps.executeUpdate();
    }

    /**
     * Upsert record.
     */
    public void upsertRecord() {
        Connection connection = null;
        try {
            connection = this.datasource.getConnection();
            ClassPathResource r = new ClassPathResource("/eu/dnetlib/templates/refreshUtilityView.sql");
            String refreshSql = IOUtils.toString(r.getInputStream());
            r = new ClassPathResource("/eu/dnetlib/templates/insertNewData.sql");
            connection.setAutoCommit(false);
            String insertNewData = IOUtils.toString(r.getInputStream());
            PreparedStatement ps = connection.prepareStatement(refreshSql);
            log.info("Refreshing View");
            ps.executeUpdate();
            log.info("Refreshing View Done!");
            ps = connection.prepareStatement(insertNewData);
            log.info("Inserting New Data");
            ps.executeUpdate();
            connection.commit();
            updateCompleteRecord(connection);

        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                if (connection != null)
                    connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
        } finally {
            closeConnection(connection);
        }

    }


    public List<DLIDatasourceInfo> listDatasources() {
        Connection connection = null;
        try {
            connection = datasource.getConnection();
            final String query = "SELECT  name, web, icon_uri, ds_id, latitude, longitude FROM dli_datasource WHERE exportable=TRUE and typology<>'resolver'";

            PreparedStatement ps = connection.prepareStatement(query);

            ResultSet resultSet = ps.executeQuery();

            List<DLIDatasourceInfo> result = Lists.newArrayList();

            while (resultSet.next()) {
                final String res_name = resultSet.getString(1);
                final String res_web = resultSet.getString(2);
                final String res_icon = resultSet.getString(3);
                final String res_id = resultSet.getString(4);
                final float latitude = resultSet.getFloat(5);
                final float longitude = resultSet.getFloat(6);

                DLIDatasourceInfo currentDatasource = new DLIDatasourceInfo();
                currentDatasource.setOfficialName(res_name);
                currentDatasource.setNamespacePrefix(res_id);
                currentDatasource.setIconURI(res_icon);
                currentDatasource.setWebSite(res_web);
                currentDatasource.setLatitude(latitude);
                currentDatasource.setLongitude(longitude);
                result.add(currentDatasource);


            }
            return result;
        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
            return null;
        } finally {
            closeConnection(connection);
        }
    }

    public DLIDatasourceInfo getDataSourceInfo(final String name, final String id) {
        Connection connection = null;
        if (StringUtils.isEmpty(name) && StringUtils.isEmpty(id))
            return null;
        String query = null;
        String argumentLookup = null;
        if (name != null) {
            query = "SELECT  name, web, icon_uri, ds_id, latitude, longitude FROM dli_datasource WHERE name = ?";
            argumentLookup = name;
        }
        if (id != null) {
            query = "SELECT  name, web, icon_uri, ds_id, latitude, longitude FROM dli_datasource WHERE ds_id = ?";
            argumentLookup = id;
        }
        try {
            connection = datasource.getConnection();
            final PreparedStatement preparedStatement = connection.prepareStatement(query);
            preparedStatement.setString(1, argumentLookup);
            ResultSet rs = preparedStatement.executeQuery();
            DLIDatasourceInfo info = new DLIDatasourceInfo();
            if (rs.next()) {
                final String res_name = rs.getString(1);
                final String res_web = rs.getString(2);
                final String res_icon = rs.getString(3);
                final String res_id = rs.getString(4);
                final float latitude = rs.getFloat(5);
                final float longitude = rs.getFloat(6);
                info.setIconURI(res_icon);
                info.setOfficialName(res_name);
                info.setWebSite(res_web);
                info.setNamespacePrefix(res_id);
                info.setLatitude(latitude);
                info.setLongitude(longitude);
                return info;

            } else return null;


        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
            return null;
        } finally {
            closeConnection(connection);
        }
    }


    /**
     * Upsert record.
     */
    public void deductRelation() {
        Connection connection = null;
        try {
            connection = this.datasource.getConnection();
            ClassPathResource r = new ClassPathResource("/eu/dnetlib/templates/DeductRelation.sql");
            String deductSQL = IOUtils.toString(r.getInputStream());
            connection.setAutoCommit(false);
            PreparedStatement ps = connection.prepareStatement(deductSQL);
            log.info("Start Deduction of Relation");
            ps.executeUpdate();
            log.info("Deduction of Relation Done!");
            connection.commit();

        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
        } finally {
            closeConnection(connection);
        }

    }

    /**
     * Gets the data source by name maps.
     *
     * @return the data source by name maps
     */
    public Map<String, Pair<String, Boolean>> getDataSourceByIDMaps() {
        Connection connection = null;
        try {
            connection = datasource.getConnection();
            connection.setAutoCommit(false);
            PreparedStatement ps = connection.prepareStatement("select ds_id, name, exportable from dli_datasource");
            ResultSet result = ps.executeQuery();
            Map<String, Pair<String, Boolean>> datasourceMap = Maps.newHashMap();
            while (result.next()) {
                datasourceMap.put(result.getString("ds_id"), new Pair<String, Boolean>(result.getString("name"), result.getBoolean("exportable")));
            }
            return datasourceMap;
        } catch (Exception e) {
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e1);
            }
            return null;
        } finally {
            closeConnection(connection);
        }
    }



    public Map<String, String> getPublisherByNameMap() {
        Connection connection = null;
        try {
            connection = datasource.getConnection();
            connection.setAutoCommit(false);
            PreparedStatement ps = connection.prepareStatement("select name, ds_id from dli_datasource");
            ResultSet result = ps.executeQuery();
            Map<String, String> publisherMap = Maps.newHashMap();
            while (result.next()) {
                publisherMap.put(result.getString("name").toLowerCase().trim(), result.getString("ds_id"));
            }
            return publisherMap;
        } catch (Exception e) {
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e1);
            }
            return null;
        } finally {
            closeConnection(connection);
        }

    }

    /**
     * Update complete record.
     *
     * @param connection the connection
     */
    private void updateCompleteRecord(final Connection connection) {
        String sql = "select * from new_dli_object_to_upsert";
        try {
            connection.setAutoCommit(false);
            PreparedStatement ps = connection.prepareStatement(sql);
            ResultSet resultSet = ps.executeQuery();
            while (resultSet.next()) {
                final String id = resultSet.getString("dli_id");
                // final String pid = resultSet.getString("pid");
                final String date = resultSet.getString("creation_date");
                final String type = resultSet.getString("type");
                final Array titles = resultSet.getArray("title");
                final Array authors = resultSet.getArray("authors");

                updateCompleteObject(connection, id, date, type, titles, authors, "complete");
                // updateCompleteRelation(connection, type, titles, pid);
                connection.commit();
            }
        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
        } finally {
            closeConnection(connection);
        }

    }

    /**
     * Update complete relation.
     *
     * @param connection the connection
     * @param type       the type
     * @param titles     the titles
     * @param pid        the pid
     * @throws SQLException the SQL exception
     */
    private void updateCompleteRelation(final Connection connection, final String type, final Array titles, final String pid, final Array authors)
            throws SQLException {
        PreparedStatement update = connection.prepareStatement(
                "update dli_relation set target_type =?, target_title=?,  target_authors=?, relationship_completion_status=? where target_pid=?");
        update.setString(1, type);
        update.setFetchSize(100);
        String title = null;
        ResultSet rs = titles.getResultSet();
        if (rs.next()) {
            title = rs.getString(2);
        }
        update.setString(2, title);
        update.setArray(3, authors);
        update.setString(4, "complete");
        update.setString(5, pid);
        update.executeUpdate();
    }

    /**
     * Update complete object.
     *
     * @param connection the connection
     * @param id         the id
     * @param date       the date
     * @param type       the type
     * @param titles     the titles
     * @param authors    the authors
     * @param status     the status
     * @throws SQLException the SQL exception
     */
    private void updateCompleteObject(final Connection connection,
                                      final String id,
                                      final String date,
                                      final String type,
                                      final Array titles,
                                      final Array authors,
                                      final String status) throws SQLException {
        PreparedStatement update =
                connection.prepareStatement("update dli_object set type =?, title=?,authors=?, completion_status=?, creation_date=? where dli_id=?");

        update.setString(1, type);
        update.setArray(2, titles);
        update.setArray(3, authors);
        update.setString(4, status);
        update.setString(5, date);
        update.setString(6, id);
        update.executeUpdate();
    }

    /**
     * Verify if Exists a particular table.
     *
     * @param tableName the table name
     * @param conn      the conn
     * @return true, if successful
     * @throws SQLException the SQL exception
     */
    private boolean existsTable(final String tableName, final Connection conn) throws SQLException {
        DatabaseMetaData md = conn.getMetaData();
        ResultSet rs = md.getTables(null, null, tableName, null);
        while (rs.next()) {
            String tName = rs.getString(3);
            if (tableName.equals(tName)) return true;
        }
        return false;
    }

    /**
     * Gets the template table.
     *
     * @return the templateTable
     */
    public Resource getTemplateTable() {
        return templateTable;
    }

    /**
     * Sets the template table.
     *
     * @param templateTable the new template table
     */
    public void setTemplateTable(final Resource templateTable) {
        this.templateTable = templateTable;
    }

    /**
     * Drop temporary data.
     */
    public void dropTemporaryData() {
        Connection connection = null;
        try {

            String sql = "delete from dli_object_tmp";
            connection = datasource.getConnection();
            connection.setAutoCommit(false);
            log.info("delete dli_object_tmp data");
            PreparedStatement ps = connection.prepareStatement(sql);
            ps.executeUpdate();
            log.info("delete dli_relation_tmp data");
            sql = "delete from dli_relation_tmp";
            ps = connection.prepareStatement(sql);
            ps.executeUpdate();

            log.info("delete record_provenance_tmp data");
            sql = "delete from record_provenance_tmp";
            ps = connection.prepareStatement(sql);
            ps.executeUpdate();

            log.info("delete record_provenance_tmp data");
            sql = "delete from relation_provenance_tmp";
            ps = connection.prepareStatement(sql);
            ps.executeUpdate();
            connection.commit();
        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
        } finally {
            closeConnection(connection);
        }
    }

    /**
     * Update exported view.
     *
     * @throws DLIException the DLI exception
     */
    public void updateExportedView() throws DLIException {
        Connection connection = null;
        try {
            connection = this.datasource.getConnection();
            ClassPathResource r = new ClassPathResource("/eu/dnetlib/templates/refresh_export_view.sql");
            String refreshSql = IOUtils.toString(r.getInputStream());
            PreparedStatement ps = connection.prepareStatement(refreshSql);
            connection.setAutoCommit(false);
            ps.executeUpdate();
            connection.commit();
        } catch (Exception e) {
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
            throw new DLIException("Error on updating Views", e);
        } finally {
            closeConnection(connection);
        }
    }

    private void insertPidInfo(final DLIObject record, final Connection connection) throws SQLException {
        final String query = "insert into pid_tmp(pid, ds_id) values (?,?)";
        PreparedStatement statement = connection.prepareStatement(query);
        statement.setString(1, record.getPid());
        statement.setString(2, record.getDatasourceProvenance().get(0).getDatasource());
        statement.executeUpdate();
    }

    public void insertPidForIntersection(final List<DLIObject> records) {
        Connection connection = null;
        DLIObject lastRecord = null;
        try {
            connection = datasource.getConnection();
            connection.setAutoCommit(false);
            for (DLIObject input : records) {
                lastRecord = input;
                if (input.getPid() == null) return;
                insertPidInfo(input, connection);
            }
            connection.commit();
            connection.setAutoCommit(true);

        } catch (Exception e) {
            log.error("Error on executing query " + lastRecord, e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
        } finally {
            closeConnection(connection);
        }

    }

    /**
     * Insert records.
     *
     * @param records the records
     */
    public void insertRecords(final List<DLIObject> records) {
        Connection connection = null;
        DLIObject lastRecord = null;
        try {
            connection = datasource.getConnection();
            connection.setAutoCommit(false);

            for (DLIObject input : records) {
                lastRecord = input;
                // ADDING OBJECT
                if (input.getPid() == null) return;
                insertObjectInfo(input, connection);
                insertObjectProvenance(input, connection, "record_provenance_tmp");

                // ADDING RELATION
                if (input.getRelations() != null) {
                    for (DLIRelation relation : input.getRelations()) {
                        insertRelation(connection, relation);
                        insertRelationProvenance(connection, relation);
                    }
                }
            }
            connection.commit();
            connection.setAutoCommit(true);

        } catch (Exception e) {
            log.error("Error on executing query " + lastRecord, e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
        } finally {
            closeConnection(connection);
        }
    }

    /**
     * Insert relation provenance.
     *
     * @param connection the connection
     * @param relation   the relation
     * @throws SQLException the SQL exception
     */
    private void insertRelationProvenance(final Connection connection, final DLIRelation relation) throws SQLException {
        String sql = "INSERT INTO relation_provenance_tmp (ds_id, id_relation, provision_mode, completion_status) " +
                "VALUES (?,?,?,?)";
        if (relation.getRelationProvenance() == null || relation.getRelationProvenance().size() == 0) return;
        PreparedStatement statement = null;
        try {
            for (DLIProvenance provenance : relation.getRelationProvenance()) {
                statement = connection.prepareStatement(sql);
                statement.setString(1, provenance.getDatasource());
                statement.setString(2, relation.getIDRelation());
                statement.setString(3, provenance.getProvisionMode());
                statement.setString(4, provenance.getCompletionStatus());
                log.debug("executing statement " + statement.toString());
                int totalUpdate = statement.executeUpdate();
                log.debug("Updated " + totalUpdate + " record");
            }
        } catch (SQLException e) {
            log.error("Error on executing statement " + statement);
            throw new SQLException(e);
        }
    }


    /**
     * Insert object provenance.
     *
     * @param input      the input
     * @param connection the connection
     * @throws SQLException the SQL exception
     */
    private void insertObjectProvenance(final DLIObject input, final Connection connection, final String table_name) throws SQLException {
        String sql = "INSERT INTO " + table_name + " (ds_id, dli_id, provision_mode, completion_status, contribution, publisher) " +
                "VALUES (?,?,?,?,?, ?)";
        if (input.getDatasourceProvenance() == null || input.getDatasourceProvenance().size() == 0) return;
        PreparedStatement statement = null;
        try {

            for (DLIProvenance provenance : input.getDatasourceProvenance()) {
                statement = connection.prepareStatement(sql);
                statement.setString(1, provenance.getDatasource());
                statement.setString(2, input.getIdentifier());
                statement.setString(3, provenance.getProvisionMode());
                statement.setString(4, provenance.getCompletionStatus());
                statement.setString(5, provenance.getDatasourceContribution());
                statement.setString(6, provenance.getPublisher());
                log.debug("executing statement " + statement.toString());
                int totalUpdate = statement.executeUpdate();
                log.debug("Updated " + totalUpdate + " record");
            }
        } catch (SQLException e) {
            log.error("Error on executing statement " + statement);
            throw new SQLException(e);
        }
    }

    /**
     * Insert relation.
     *
     * @param connection the connection
     * @param relation   the relation
     * @throws SQLException the SQL exception
     */
    private void insertRelation(final Connection connection, final DLIRelation relation) throws SQLException {
        PreparedStatement statement = null;
        try {
            if (relation.getTargetPID() == null) return;
            int totalUpdate;
            String sql_relation =
                    " INSERT INTO dli_relation_tmp (id_relation, source_record, target_type, target_pid, target_pid_type, target_title, relation_semantic, relationship_completion_status, source_pid, target_authors)"
                            + " VALUES (?,?,?,?,?,?,?,?,?, ?)";

            statement = connection.prepareStatement(sql_relation);
            statement.setString(1, relation.getIDRelation());
            statement.setString(2, relation.getSourceRecordId());
            if (relation.getTargetType() != null) {
                statement.setString(3, relation.getTargetType().toString());
            } else {
                statement.setString(3, null);
            }
            if (relation.getTargetPID() != null) {
                statement.setString(4, relation.getTargetPID().getId());
                statement.setString(5, relation.getTargetPID().getType());
            }
            statement.setString(6, relation.getTargetTitle());
            statement.setString(7, relation.getRelationSemantics());
            statement.setString(8, relation.getCompletionStatus());
            statement.setString(9, relation.getSourcePid());

            Array authorsArray = null;

            if (relation.getAuthors() != null) {
                authorsArray = connection.createArrayOf("text", relation.getAuthors().toArray());
            }
            statement.setArray(10, authorsArray);

            log.debug("executing statement " + statement.toString());

            totalUpdate = statement.executeUpdate();

            log.debug("Updated " + totalUpdate + " record");
        } catch (SQLException e) {
            log.error("Error on executing statement " + statement);
            throw new SQLException(e);
        }
    }

    /**
     * Insert object info.
     *
     * @param input      the input
     * @param connection the connection
     * @throws SQLException the SQL exception
     */
    private void insertObjectInfo(final DLIObject input, final Connection connection) throws SQLException {
        String sql = "INSERT INTO dli_object_tmp (dli_identifier, pid, pid_type, title, authors, creation_date, type, completion_status) " +
                "VALUES (?,?,?,?,?,?,?,?)";
        PreparedStatement statement = null;
        try {
            statement = connection.prepareStatement(sql);
            statement.setString(1, input.getIdentifier());
            statement.setString(2, input.getPid());
            statement.setString(3, input.getPidType());
            statement.setArray(4, connection.createArrayOf("text", input.getTitles()));
            statement.setArray(5, connection.createArrayOf("text", input.getAuthors()));
            if (input.getDate() != null && input.getDate().length() > 50) {
                input.setDate(input.getDate().substring(0, 20));
            }
            statement.setString(6, input.getDate());
            if (input.getType() != null) {
                statement.setString(7, input.getType().toString());
            } else {
                statement.setString(7, null);
            }
            statement.setString(8, input.getCompletionStatus());
            log.debug("executing statement " + statement.toString());
            int totalUpdate = statement.executeUpdate();
            log.debug("Updated " + totalUpdate + " record");
        } catch (SQLException e) {
            log.error("Error on executing statement " + statement);
            throw new SQLException(e);
        }
    }


    public void deleteRecordOfDatasource() {

    }


    /**
     * Gets the all record.
     *
     * @return the all record
     */
    public Iterable<String> getAllRecord(final boolean exportIntersection) {
        final DLIDBManager currentManager = this;

        return new Iterable<String>() {

            @Override
            public Iterator<String> iterator() {
                final Function<DLIObject, String> convertFunction = new Function<DLIObject, String>() {
                    @Override
                    public String apply(DLIObject input) {
                        if (input.getIdentifier()== null || StringUtils.isEmpty(input.getIdentifier()))
                            log.error("OGGETTO NULLO");
                        return extractXMLfromDLIObject(input);
                    }
                };
                return new DLIDBIterator<String>(currentManager, datasource, exportIntersection, convertFunction, true);
            }
        };
    }


    public Iterable<DLIObject> getAllDLIObjectRecord(final boolean exportIntersection, final boolean limitRelation) {
        final DLIDBManager currentManager = this;

        return new Iterable<DLIObject>() {

            @Override
            public Iterator<DLIObject> iterator() {
                final Function<DLIObject, DLIObject> convertFunction = new Function<DLIObject, DLIObject>() {
                    @Override
                    public DLIObject apply(DLIObject input) {
                        return input;
                    }
                };
                return new DLIDBIterator<DLIObject>(currentManager, datasource, exportIntersection, convertFunction, limitRelation);
            }
        };
    }


    /**
     * Extract xm lfrom dli object.
     *
     * @param object the object
     * @return the string
     */
    public String extractXMLfromDLIObject(final DLIObject object) {
        if (object != null) {
            this.xmlTemplate.removeAttribute("object");
            this.xmlTemplate.setAttribute("object", object);
            return xmlTemplate.toString();
        }
        return null;
    }

    /**
     * Sets the template.
     *
     * @param template the new template
     * @throws IOException Signals that an I/O exception has occurred.
     */
    @Required
    public void setTemplate(final Resource template) throws IOException {

        xmlTemplate = new StringTemplate(IOUtils.toString(template.getInputStream()));
    }

    /**
     * Gets the unresolved records.
     *
     * @return the unresolved records
     */
    public Iterable<DLIObject> getUnresolvedRecords() {

        String sql = "select pid, pid_type from dli_object where completion_status <> ?";
        Connection connection = null;
        try {
            connection = datasource.getConnection();
            final Connection inputConnection = connection;
            PreparedStatement ps = connection.prepareStatement(sql);
            ps.setString(1, DLICompletionStatus.complete.toString());
            final ResultSet unresolvedResultSet = ps.executeQuery();

            return new Iterable<DLIObject>() {

                @Override
                public Iterator<DLIObject> iterator() {

                    return new Iterator<DLIObject>() {

                        DLIObject nextObject;

                        {
                            nextObject = constructNext();
                        }

                        private DLIObject constructNext() {
                            try {
                                if (unresolvedResultSet.next() == false) {
                                    tryToCloseConnection();
                                    return null;
                                }
                                String pid = unresolvedResultSet.getString("pid");
                                String pidType = unresolvedResultSet.getString("pid_type");

                                DLIObject object = new DLIObject();
                                object.setPid(pid);
                                object.setPidType(pidType);
                                return object;

                            } catch (SQLException e) {
                                return null;
                            }
                        }

                        private void tryToCloseConnection() {
                            if (inputConnection != null) {
                                try {
                                    inputConnection.close();
                                } catch (SQLException e1) {
                                    log.error("Error on closing connection", e1);
                                }
                            }
                        }

                        @Override
                        public boolean hasNext() {
                            return nextObject != null;
                        }

                        @Override
                        public DLIObject next() {
                            DLIObject tmp = nextObject;
                            nextObject = constructNext();
                            return tmp;
                        }

                        @Override
                        public void remove() {
                            // TODO Auto-generated method stub

                        }
                    };
                }
            };

        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
            return null;
        }
    }

    /**
     * Gets the stats.
     *
     * @return the stats
     */
    public HashMap<String, StatsInfo> getStats() {

        String getAllObjectSQL =
                "select count(*) as total, d.ds_id as datasource from dli_object o left outer join record_provenance r on o.dli_id= r.dli_id left outer join dli_datasource d on r.ds_id = d.ds_id  where d.exportable= true group by d.ds_id";
        String getAllObjectByTypeSQL =
                "select count(*) as total, d.ds_id as datasource from dli_object o left outer join record_provenance r on o.dli_id= r.dli_id left outer join dli_datasource d on r.ds_id = d.ds_id   where o.type=? and d.exportable= true group by d.ds_id";
        String getAllRelationByTypeSQL =
                "select count(*) as total, rp.ds_id as datasource from dli_object o left outer join dli_relation r on o.dli_id = r.source_record left outer join relation_provenance rp on r.id_relation = rp.id_relation  where o.type=? and r.target_type=? group by rp.ds_id";
        String getTotalRelationSQL = "select count(*) as total , ds_id as datasource from relation_provenance group by ds_id";

        Connection connection = null;
        try {
            connection = datasource.getConnection();

            HashMap<String, StatsInfo> stats = Maps.newHashMap();
            Map<String, Pair<String, Boolean>> datasourceMap = getDataSourceByIDMaps();

            PreparedStatement ps = connection.prepareStatement(getAllObjectByTypeSQL);
            ps.setString(1, DLIObjectType.publication.toString());

            ResultSet result = ps.executeQuery();
            while (result.next()) {
                int total = result.getInt("total");
                String ds = result.getString("datasource");

                String nameDS = datasourceMap.get(ds).getKey();
                if (datasourceMap.get(ds).getValue()) {
                    if (!stats.containsKey(nameDS)) {
                        stats.put(nameDS, new StatsInfo(nameDS));
                    }
                    StatsInfo currentStats = stats.get(nameDS);
                    currentStats.setAcronym(ds);
                    currentStats.setNumberOfPublication(total);
                }
            }

            ps = connection.prepareStatement(getAllObjectSQL);

            result = ps.executeQuery();
            while (result.next()) {
                int total = result.getInt("total");
                String ds = result.getString("datasource");

                String nameDS = datasourceMap.get(ds).getKey();
                if (!stats.containsKey(nameDS)) {
                    stats.put(nameDS, new StatsInfo(nameDS));
                }
                StatsInfo currentStats = stats.get(nameDS);
                currentStats.setAcronym(ds);
                currentStats.setNumberOfObjects(total);
            }

            ps = connection.prepareStatement(getAllObjectByTypeSQL);
            ps.setString(1, DLIObjectType.dataset.toString());

            result = ps.executeQuery();
            while (result.next()) {
                int total = result.getInt("total");
                String ds = result.getString("datasource");

                String nameDS = datasourceMap.get(ds).getKey();
                if (!stats.containsKey(nameDS)) {
                    stats.put(nameDS, new StatsInfo(nameDS));
                }
                StatsInfo currentStats = stats.get(nameDS);
                currentStats.setAcronym(ds);
                currentStats.setNumberOfDatasets(total);
            }

            ps = connection.prepareStatement(getAllRelationByTypeSQL);
            ps.setString(1, DLIObjectType.dataset.toString());
            ps.setString(2, DLIObjectType.dataset.toString());

            result = ps.executeQuery();
            while (result.next()) {
                int total = result.getInt("total");
                String ds = result.getString("datasource");

                String nameDS = datasourceMap.get(ds).getKey();
                if (!stats.containsKey(nameDS)) {
                    stats.put(nameDS, new StatsInfo(nameDS));
                }
                StatsInfo currentStats = stats.get(nameDS);
                currentStats.setDatasetToDataset(total);
                currentStats.setAcronym(ds);
            }

            ps = connection.prepareStatement(getAllRelationByTypeSQL);
            ps.setString(1, DLIObjectType.dataset.toString());
            ps.setString(2, DLIObjectType.publication.toString());

            result = ps.executeQuery();
            while (result.next()) {
                int total = result.getInt("total");
                String ds = result.getString("datasource");

                String nameDS = datasourceMap.get(ds).getKey();
                if (!stats.containsKey(nameDS)) {
                    stats.put(nameDS, new StatsInfo(nameDS));
                }
                StatsInfo currentStats = stats.get(nameDS);
                currentStats.setDatasetToPublication(total);
                currentStats.setAcronym(ds);
            }

            ps = connection.prepareStatement(getTotalRelationSQL);

            result = ps.executeQuery();
            while (result.next()) {
                int total = result.getInt("total");
                String ds = result.getString("datasource");

                String nameDS = datasourceMap.get(ds).getKey();
                if (!stats.containsKey(nameDS)) {
                    stats.put(nameDS, new StatsInfo(nameDS));
                }
                StatsInfo currentStats = stats.get(nameDS);
                currentStats.setNumberOfRelations(total);
                currentStats.setAcronym(ds);
            }

            return stats;
        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
            return null;
        }

    }

    /**
     * Update resolved.
     *
     * @param resolvedbuffer the resolvedbuffer
     */
    public void updateResolved(final List<DLIObject> resolvedbuffer) {
        Connection connection = null;
        try {
            connection = datasource.getConnection();
            connection.setAutoCommit(false);

            for (DLIObject input : resolvedbuffer) {
                updateCompleteObject(connection, input.getIdentifier(), input.getDate(), input.getType().toString(),
                        connection.createArrayOf("text", input.getTitles()), connection.createArrayOf("text", input.getAuthors()),
                        input.getCompletionStatus());
                insertObjectProvenance(input, connection, "record_provenance");
                updateCompleteRelation(connection, input.getType().toString(), connection.createArrayOf("text", input.getTitles()), input.getPid(),
                        connection.createArrayOf("text", input.getAuthors()));
            }
            connection.commit();

        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
        } finally {
            closeConnection(connection);
        }

    }

    /**
     * Update unresolved.
     *
     * @param unresolvedbuffer the unresolvedbuffer
     */
    public void updateUnresolved(final List<DLIObject> unresolvedbuffer) {
        Connection connection = null;
        try {

        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
        } finally {
            closeConnection(connection);
        }

    }


    public void deleteOldPid(final String datasourcePrefix) throws SQLException {
        Connection connection = null;
        try {
            connection = datasource.getConnection();
            connection.setAutoCommit(false);
            String query = "delete from record_provenance where ds_id=?";
            PreparedStatement preparedStatement = connection.prepareStatement(query);
            preparedStatement.setString(1, datasourcePrefix);
            preparedStatement.executeUpdate();
            preparedStatement = connection.prepareStatement("delete from pid_tmp;");
            preparedStatement.executeUpdate();
            connection.commit();
            connection.setAutoCommit(true);

        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
        } finally {
            closeConnection(connection);
        }
    }

    private void closeConnection(final Connection connection) {
        if (connection != null) {
            try {
                connection.close();
            } catch (SQLException e) {
                log.error("Erron on closing connection", e);
            }
        }
    }

    public void insertIntersectedPid() {
        Connection connection = null;
        try {
            connection = datasource.getConnection();
            connection.setAutoCommit(false);
            String query =
                    "insert into record_provenance(dli_id, ds_id, provision_mode, completion_status, contribution) select d.dli_id, p.ds_id, 'collected', 'incomplete', array['pid'] from dli_object d  join pid_tmp p on d.pid = p.pid;";
            PreparedStatement preparedStatement = connection.prepareStatement(query);
            preparedStatement.executeUpdate();
            connection.commit();
            connection.setAutoCommit(true);

        } catch (Exception e) {
            log.error("Error on executing query ", e);
            try {
                connection.rollback();
            } catch (SQLException e1) {
                log.error("Error on rollback ", e);
            }
        } finally {
            closeConnection(connection);
        }

    }

    public void setDatasource(final BasicDataSource datasource) {
        this.datasource = datasource;

    }

}
