/**
 *
 */
package eu.dnetlib.msro.workflows.dli.manager;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import eu.dnetlib.miscutils.collections.Pair;
import eu.dnetlib.msro.workflows.dli.model.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.sql.DataSource;
import java.sql.*;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * @author sandro
 */
public class DLIDBIterator<T> implements Iterator<T> {

    private static final Log log = LogFactory.getLog(DLIDBIterator.class); // NOPMD by marko on 11/24/08 5:02 PM
    private static final int MAX_ELEMENT = 3000;
    private final String sqlString = "select * from exported_view";
    private final DLIDBManager manager;
    private final DataSource inputDatasource;
    private final boolean limitRelation;
    private DLIObject nextObject;
    private Connection currentConnection;
    private ResultSet inputResutlSet;
    private DLIObject dliObject;
    private Map<String, Pair<String, Boolean>> datasourceMap;
    private Map<String, String> publisherByNameMap;
    private Function<DLIObject, T> converter;
    private boolean exportIntersection = false;

    /**
     *
     */
    public DLIDBIterator(final DLIDBManager manager, final DataSource inputDatasource, final boolean exportIntersection, final Function<DLIObject, T> converter, final boolean limitRelation) {
        this.manager = manager;
        this.inputDatasource = inputDatasource;
        this.exportIntersection = exportIntersection;
        this.datasourceMap = manager.getDataSourceByIDMaps();
        this.publisherByNameMap = manager.getPublisherByNameMap();
        this.limitRelation = limitRelation;
        this.converter = converter;
        nextObject = constructNextObject();


    }

    /**
     * {@inheritDoc}
     *
     * @see java.util.Iterator#hasNext()
     */
    @Override
    public boolean hasNext() {
        if (nextObject == null) {
            tryToCloseConnection();
        }
        return nextObject != null;
    }

    /**
     * {@inheritDoc}
     *
     * @see java.util.Iterator#next()
     */
    @Override
    public T next() {
        DLIObject tmp = nextObject;
        nextObject = constructNextObject();
        return converter.apply(tmp);
    }

    /**
     * {@inheritDoc}
     *
     * @see java.util.Iterator#remove()
     */
    @Override
    public void remove() {
        // TODO Auto-generated method stub

    }

    private DLIObject constructNextObject() {

        if (inputResutlSet == null) {
            createInputResultSet();
        }
        try {
            if (this.inputResutlSet == null || inputResutlSet.next() == false) {
                tryToCloseConnection();
                return null;
            }
            dliObject = new DLIObject();
            dliObject.setPid(inputResutlSet.getString("pid"));
            dliObject.setPidType(inputResutlSet.getString("pid_type"));

            List<String> titles = Lists.newArrayList();
            Array tArray = inputResutlSet.getArray("title");
            if (tArray != null) {
                ResultSet trs = tArray.getResultSet();
                while (trs.next()) {
                    titles.add(trs.getString(2));
                }
            }
            dliObject.setTitles(titles.toArray(new String[0]));
            List<String> authors = Lists.newArrayList();
            Array aArray = inputResutlSet.getArray("authors");
            if (aArray != null) {
                ResultSet trs = aArray.getResultSet();
                while (trs.next()) {
                    authors.add(trs.getString(2));
                }
            }
            dliObject.setAuthors(authors.toArray(new String[0]));
            dliObject.setDate(inputResutlSet.getString("creation_date"));
            String type = inputResutlSet.getString("type");
            try {
                dliObject.setType(DLIObjectType.valueOf(type));
            } catch (Throwable e) {
            }
            dliObject.setCompletionStatus(inputResutlSet.getString("completion_status"));
            String provenance = inputResutlSet.getString("provenance");
            List<DLIProvenance> prov = parseProvenance(provenance);
            dliObject.setDatasourceProvenance(prov);
            String s = inputResutlSet.getString("relations");
            try {
                if (s != null) {
                    List<DLIRelation> rels = parseRelation(s);
                    if (rels != null) {
                        dliObject.setRelations(rels);
                    }
                }
            } catch (Exception e) {
                log.error("Error for DOI " + inputResutlSet.getString(2) + "   " + s, e);
            }
            //manager.extractXMLfromDLIObject(dliObject)
            return dliObject;
        } catch (Exception e) {
            log.error("Error on iterating on all objects", e);
            return null;
        }
    }

    private void createInputResultSet() {
        try {
            currentConnection = this.inputDatasource.getConnection();
            PreparedStatement st = currentConnection.prepareStatement(sqlString);
            currentConnection.setAutoCommit(false);
            st.setFetchSize(100);
            this.inputResutlSet = st.executeQuery();
        } catch (Exception e) {
            log.error("Error on iterating on all objects", e);
            if (currentConnection != null) {
                tryToCloseConnection();
            }
        }
    }

    private void tryToCloseConnection() {
        if (currentConnection != null) {
            try {
                currentConnection.close();
            } catch (SQLException e1) {
                log.error("Error on closing connection", e1);
            }
        }

    }

    private String getObject(final JsonElement element, final String key) {
        if (element.isJsonNull()) return null;
        if (element.getAsJsonObject().get(key).isJsonNull()) return null;
        return element.getAsJsonObject().get(key).getAsString();
    }

    private List<String> getArrayObject(final JsonElement element, final String key) {
        if (element.isJsonNull()) return null;
        if (element.getAsJsonObject().get(key).isJsonNull()) return null;
        JsonArray inputArray = element.getAsJsonObject().get(key).getAsJsonArray();
        if (inputArray == null || inputArray.size() == 0)
            return null;
        int size = inputArray.size();
        List<String> result = Lists.newArrayList();
        for (int i = 0; i < size; i++) {
            result.add(inputArray.get(i).getAsString());
        }
        return result;

    }

    private List<DLIProvenance> parseProvenance(final String input) {
        if (input == null) return null;
        String fixedInput = input.replaceAll("\\[", "").replaceAll("\\]", "");

        JsonElement jElement = new JsonParser().parse("[" + fixedInput + "]");
        if (jElement.isJsonNull()) return null;
        JsonArray datasourcesArray = jElement.getAsJsonArray();
        if (datasourcesArray.size() == 0)
            return null;
        Map<String, DLIProvenance> provenanceMaps = Maps.newHashMap();
        for (JsonElement currentDatasource : datasourcesArray) {
            String ds_id = getObject(currentDatasource, "ds_id");
            String provision_mode = getObject(currentDatasource, "provision_mode");
            String completion_status_prov = getObject(currentDatasource, "completion_status");
            String contribution = getObject(currentDatasource, "contribution");
            String collection_date = getObject(currentDatasource, "collection_date");
            String publisher = getObject(currentDatasource, "publisher");
            String publisherId = null;
            if(publisher!=null && !"null".equals(publisher)) {
                if(publisherByNameMap.containsKey(publisher.toLowerCase().trim())) {
                    publisherId = publisherByNameMap.get(publisher.toLowerCase().trim());
                }
            }
            Pair<String, Boolean> ds_info = datasourceMap.get(ds_id);
            ds_id = ds_info.getKey();
            final boolean visible = ds_info.getValue();
            if (visible || exportIntersection) {
                provenanceMaps.put(ds_id, new DLIProvenance(ds_id, provision_mode, completion_status_prov, contribution, collection_date, visible,publisherByNameMap.get(ds_id.toLowerCase().trim()), publisher, publisherId));
            }

        }
        return Lists.newArrayList(provenanceMaps.values());
    }

    private List<DLIRelation> parseRelation(final String input) {
        if (input == null) return null;

        List<DLIRelation> result = Lists.newArrayList();

        JsonElement jElement = new JsonParser().parse(input);
        JsonArray jarray = jElement.getAsJsonArray();

        int i = 0;

        for (JsonElement relation : jarray) {

            DLIRelation currentrelation = new DLIRelation();
            String record_id = getObject(relation, "record_id");
            String source_pid = getObject(relation, "source_pid");
            String target_type = getObject(relation, "target_type");
            String target_pid = getObject(relation, "target_pid");
            String target_pid_type = getObject(relation, "target_pid_type");

            String title = getObject(relation, "title");
            List<String> authors = getArrayObject(relation, "authors");
            String relationSemantics = getObject(relation, "relation");
            String completion_status = getObject(relation, "completion_status");

            currentrelation.setSourceRecordId(record_id);
            currentrelation.setSourcePid(source_pid);
            try {
                currentrelation.setTargetType(DLIObjectType.valueOf(target_type));
            } catch (Throwable e) {

            }
            currentrelation.setTargetPID(new DLIPID(target_pid, target_pid_type));
            currentrelation.setTargetTitle(title);
            currentrelation.setRelationSemantics(relationSemantics);
            currentrelation.setCompletionStatus(completion_status);
            currentrelation.setAuthors(authors);

            if (relation.isJsonNull() == false) {
                JsonElement datasourceJson = relation.getAsJsonObject().get("datasources");
                if (!datasourceJson.isJsonNull()) {

                    List<DLIProvenance> prov = Lists.newArrayList();
                    JsonArray datasourcesArray = datasourceJson.getAsJsonArray();

                    for (JsonElement currentDatasource : datasourcesArray) {
                        String ds_id = getObject(currentDatasource, "ds_id");
                        String provision_mode = getObject(currentDatasource, "provision_mode");
                        String completion_status_prov = getObject(currentDatasource, "completion_status");
                        String collection_date = getObject(currentDatasource, "collection_date");
                        Pair<String, Boolean> ds_info = datasourceMap.get(ds_id);
                        ds_id = ds_info.getKey();
                        final boolean visible = ds_info.getValue();
                        prov.add(new DLIProvenance(ds_id, provision_mode, completion_status_prov, null, collection_date, visible));
                    }

                    currentrelation.setRelationProvenance(prov);

                }
            }
            result.add(currentrelation);
            if (i++ > MAX_ELEMENT && limitRelation == true) {


                return result;
            }
        }
        return result;


    }

}
