package eu.dnetlib.msro.dli.workflows.nodes.feeding;

import com.google.common.collect.Lists;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;
import eu.dnetlib.enabling.resultset.IterableResultSetFactory;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.enabling.resultset.client.utils.EPRUtils;
import eu.dnetlib.msro.dli.workflows.nodes.feeding.utility.DLIRecordParser;
import eu.dnetlib.msro.workflows.dli.manager.DLIDBManager;
import eu.dnetlib.msro.workflows.dli.model.DLIObject;
import eu.dnetlib.msro.workflows.nodes.ProgressJobNode;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.util.ProgressProvider;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import javax.xml.ws.wsaddressing.W3CEndpointReference;
import java.util.List;

/**
 * The Class FeedPIDJobNode is the workflow node responsible to feed
 * pid for intersection
 */
public class FeedPIDJobNode extends SimpleJobNode implements ProgressJobNode {

    /**
     * The Constant log.
     */
    private static final Log log = LogFactory.getLog(FeedPIDJobNode.class);

    /**
     * The input epr.
     */
    private String inputEprParam;

    /**
     * The result set client factory.
     */
    @Autowired
    private ResultSetClientFactory resultSetClientFactory;

    /**
     * The result set factory.
     */
    @Resource(name = "iterableResultSetFactory")
    private IterableResultSetFactory resultSetFactory;

    private int currentRecord = 0;

    /**
     * The db manager.
     */
    @Autowired
    private DLIDBManager dbManager;

    /**
     * {@inheritDoc}
     *
     * @see eu.dnetlib.msro.workflows.nodes.SimpleJobNode#execute(com.googlecode.sarasvati.NodeToken)
     */
    @Override
    protected String execute(final NodeToken token) throws Exception {

        final W3CEndpointReference inputEpr = new EPRUtils().getEpr(token.getEnv().getAttribute(getInputEprParam()));
        final Iterable<String> inputPid = resultSetClientFactory.getClient(inputEpr);
        DLIRecordParser parser = new DLIRecordParser();
        List<DLIObject> records = Lists.newArrayList();
        for (String inputRecord : inputPid) {
            currentRecord++;
            DLIObject inputObject = parser.parseRecord(inputRecord);
            if (inputObject.getPid() != null && inputObject.getPid().length() > 0) {
                records.add(inputObject);
            }
            if (records.size() > 1000) {
                dbManager.insertPidForIntersection(records);
                records.clear();
            }

        }
        dbManager.insertPidForIntersection(records);

        return Arc.DEFAULT_ARC;
    }

    /**
     * {@inheritDoc}
     *
     * @see eu.dnetlib.msro.workflows.nodes.ProgressJobNode#getProgressProvider()
     */
    @Override
    public ProgressProvider getProgressProvider() {

        return new ProgressProvider() {

            @Override
            public boolean isInaccurate() {
                return true;
            }

            @Override
            public int getTotalValue() {
                return 0;
            }

            @Override
            public int getCurrentValue() {
                return currentRecord;
            }
        };
    }

    public String getInputEprParam() {
        return inputEprParam;
    }

    public void setInputEprParam(final String inputEprParam) {
        this.inputEprParam = inputEprParam;
    }

}
