/**
 *
 */
package eu.dnetlib.msro.dli.workflows.nodes.feeding;

import com.google.common.collect.Lists;
import eu.dnetlib.msro.dli.workflows.nodes.feeding.utility.DLIRecordParser;
import eu.dnetlib.msro.dli.workflows.nodes.transform.DLIUtils;
import eu.dnetlib.msro.workflows.dli.manager.DLIDBManager;
import eu.dnetlib.msro.workflows.dli.model.*;
import eu.dnetlib.msro.workflows.dli.publisher.PublisherConfigurationParameters;
import eu.dnetlib.msro.workflows.dli.publisher.PublisherResolver;
import eu.dnetlib.msro.workflows.dli.publisher.PublisherResolverRule;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;

/**
 * The Class DatabaseFeederWorker.
 *
 * @author sandro
 */
public class DatabaseFeederWorker implements Callable<Boolean> {

    /**
     * The Constant log.
     */
    private static final Log log = LogFactory.getLog(DatabaseFeederWorker.class); // NOPMD by marko on 11/24/08 5:02 PM

    private final static String patternDOI = "\\b(10[.][0-9]{4,}(?:[.][0-9]+)*/(?:(?![\"&(\\'<>])\\S)+)\\b";

    /**
     * The input queue.
     */
    private final BlockingQueue<String> inputQueue;

    /**
     * The parser.
     */
    private final DLIRecordParser parser;

    /**
     * The manager.
     */
    private final DLIDBManager manager;

    /**
     * The namespace.
     */
    private final String namespace;

    /**
     * The buffer.
     */
    private final List<DLIObject> buffer;

    /**
     * The terminator queue.
     */
    private final String terminatorQueue;

    private final PublisherResolver publisherResolver;

    /**
     * Instantiates a new enrich feeder.
     *
     * @param inputQueue      the input queue
     * @param parser          the parser
     * @param manager         the manager
     * @param namespace       the namespace
     * @param terminatorQueue the terminator queue
     */
    public DatabaseFeederWorker(final BlockingQueue<String> inputQueue, final DLIRecordParser parser, final DLIDBManager manager, final String namespace,
                                final String terminatorQueue, final PublisherResolver resolver) {
        super();
        this.inputQueue = inputQueue;
        this.parser = parser;
        this.manager = manager;
        this.namespace = namespace;
        this.terminatorQueue = terminatorQueue;
        this.buffer = Lists.newArrayList();
        this.publisherResolver = resolver;
    }

    /**
     * {@inheritDoc}
     *
     * @see java.util.concurrent.Callable#call()
     */
    @Override
    public Boolean call() throws Exception {

        String nextRecord = null;
        // TAKE THE FIRST ELEMENT AND CHECK IF IS NOT THE
        // END ELEMENT
        try {
            nextRecord = this.inputQueue.take();
            if (nextRecord == this.terminatorQueue) {
                log.debug(":Found terminator record");
                this.inputQueue.put(terminatorQueue);
                return true;
            }
        } catch (InterruptedException e) {
            log.error("Error on taking an element on queue", e);
        }
        while (nextRecord != null && nextRecord != terminatorQueue) {
            try {
                final DLIObject extractedObject = parser.parseRecord(nextRecord);
                if (extractedObject != null) {
                    if (extractedObject.getDatasourceProvenance() != null) {
                        for (DLIProvenance prov : extractedObject.getDatasourceProvenance()) {
                            extractedObject.fixContribution(prov);
                        }
                    }
                    if (extractedObject.getCompletionStatus() == null) {
                        extractedObject.setCompletionStatus(DLICompletionStatus.complete.toString());
                    }
                    fixRecordProvenance(extractedObject);
                    fixRecordPublisher(extractedObject, publisherResolver.getParameters().getSource());
                    normalizeDOI(extractedObject);
                    if (extractRelation(extractedObject)) {
                        buffer.add(extractedObject);
                    }
                    if (buffer.size() > 1000) {
                        try {
                            manager.insertRecords(buffer);
                            buffer.clear();
                        } catch (Throwable e) {
                            log.error("Error on insert the new batch of data", e);
                        }
                    }
                }
                // EXTRACTING NEXT ELEMENT ON THE QUEUE
                nextRecord = this.inputQueue.take();
                log.debug(":Taken item");
                // IF THE ELEMENT IS THE LAST THEN COMPLETE
                if (nextRecord == this.terminatorQueue) {
                    manager.insertRecords(buffer);
                    log.info("Found terminator record");
                    return true;
                }
            } catch (Throwable e) {
                log.error("Error on enriching record", e);
                return false;
            }
        }
        return true;
    }

    private void fixRecordPublisher(final DLIObject extractedObject, final List<PublisherResolverRule> source) {
        if (publisherResolver== null || publisherResolver.getParameters()==null)
            return;
        if(source !=null && source.size()>0){
            for (PublisherResolverRule item : source ) {
                if(item.getField()==null) {
                    extractedObject.getDatasourceProvenance().get(0).setPublisher(item.getKey());
                }
                else {
                    String inputValue = "";
                    if (PublisherResolver.PID_TYPE.equals(item.getField())) {
                        inputValue = extractedObject.getPidType();
                    }

                    if (PublisherResolver.OBJECT_TYPE.equals(item.getField())) {
                        inputValue = extractedObject.getType().toString();
                    }
                    if (PublisherResolver.PID.equals(item.getField())) {
                        inputValue = extractedObject.getPid();
                    }
                    if(checkConditionForPublisher(inputValue, item.getRegExp(), extractedObject, item.getKey())== true)
                        return;
                }
            }
        }
    }



    private boolean checkConditionForPublisher(final String value, final String regExp, final DLIObject inputObject, final String key) {
        if(value== null)
            return false;
        if(value.toLowerCase().trim().matches(regExp)) {
            inputObject.getDatasourceProvenance().get(0).setPublisher(key);
            return true;
        }
        return  false;
    }


    /**
     * Extract relation.
     *
     * @param input the input
     */
    private boolean extractRelation(final DLIObject input) {
        boolean addedOne = false;
        List<DLIRelation> relations = input.getRelations();

        if (relations != null) {
            for (DLIRelation relation : relations) {
                DLIObject currentObject = extractRecordFromRelation(input, relation);
                if (currentObject != null) {
                    fixRecordPublisher(currentObject, publisherResolver.getParameters().getTarget());
                    buffer.add(currentObject);
                    addedOne = true;
                }
            }
        }
        return addedOne;
    }

    /**
     * Fix relation status.
     *
     * @param relation the relation
     */
    private void fixRelationStatus(final DLIRelation relation) {
        relation.setCompletionStatus(DLICompletionStatus.complete.toString());
        if (relation.getRelationProvenance() == null || relation.getRelationProvenance().size() == 0) {
            DLIProvenance prov =
                    new DLIProvenance(namespace, DLIProvisionMode.collected.toString(), DLICompletionStatus.incomplete.toString(), null, null, true);
            relation.setRelationProvenance(Lists.newArrayList(prov));
        }
    }

    private boolean isPidDOI(final String pid) {
        if (pid == null) return false;
        final String tmpPid = pid.replace("http://dx.doi.org/", "");
        return tmpPid.matches(patternDOI);

    }

    /**
     * Extract record from relation.
     *
     * @param input    the input
     * @param relation the relation
     * @return the DLI object
     */
    private DLIObject extractRecordFromRelation(final DLIObject input, final DLIRelation relation) {
        DLIPID currentPIDRelation = relation.getTargetPID();

        if (currentPIDRelation == null)
            return null;
        if (isPidDOI(currentPIDRelation.getId()) && !"doi".equals(currentPIDRelation.getType())) {
            currentPIDRelation.setType("doi");
        }
        // CREATE THE NEW OBJECT
        DLIObject dbObject = new DLIObject();
        // SET THE PID
        dbObject.setPid(currentPIDRelation.getId());
        dbObject.setPidType(currentPIDRelation.getType());
        dbObject.setCompletionStatus(DLICompletionStatus.incomplete.toString());

        // SET THE PROVENANCE STATUS INCOMPLETE, PROVISION MODE COLLECTED
        DLIProvenance dbObjectProvenance =
                new DLIProvenance(input.getDatasourceProvenance().get(0).getDatasource(), DLIProvisionMode.collected.toString(), DLICompletionStatus.incomplete.toString(), null, null, true);
        dbObject.setDatasourceProvenance(Lists.newArrayList(dbObjectProvenance));
        dbObject.fixContribution(dbObjectProvenance);

        // CREATE RELATION FROM NEW OBJECT TO INPUT
        DLIRelation objectToInput = new DLIRelation();

        objectToInput.setSourcePid(dbObject.getPid());
        objectToInput.setSourceRecordId(dbObject.getIdentifier());
        objectToInput.setTargetPID(new DLIPID(input.getPid(), input.getPidType()));
        objectToInput.setTargetType(input.getType());
        objectToInput.setCompletionStatus(dbObject.getCompletionStatus());
        objectToInput.setRelationSemantics(DLIUtils.getInverse(relation.getRelationSemantics()));
        if (input.getTitles() != null && input.getTitles().length > 0) {
            objectToInput.setTargetTitle(input.getTitles()[0]);
        }
        if (input.getAuthors() != null) {
            List<String> authors = Lists.newArrayList();
            for (String s : input.getAuthors()) {
                authors.add(s);
            }
            objectToInput.setAuthors(authors);
        }
        DLIProvenance objectToInputProvenance =
                new DLIProvenance(namespace, DLIProvisionMode.collected.toString(), input.getCompletionStatus(), null, null, true);
        objectToInput.setRelationProvenance(Lists.newArrayList(objectToInputProvenance));
        objectToInput.setCompletionStatus(input.getCompletionStatus());
        dbObject.setRelations(Lists.newArrayList(objectToInput));

        // UPDATE RELATION FROM INPUT TO NEW OBJECT
        relation.setSourcePid(input.getPid());
        relation.setSourceRecordId(input.getIdentifier());
        relation.setCompletionStatus(DLICompletionStatus.incomplete.toString());
        if (dbObject.getAuthors() != null) {
            List<String> authors = Lists.newArrayList();
            for (String s : dbObject.getAuthors()) {
                authors.add(s);
            }
            relation.setAuthors(authors);
        }
        DLIProvenance inputToDbObjectProvenance =
                new DLIProvenance(namespace, DLIProvisionMode.collected.toString(), DLICompletionStatus.incomplete.toString(), null, null, true);
        relation.setRelationProvenance(Lists.newArrayList(inputToDbObjectProvenance));

        return dbObject;

    }

    /**
     * Because a relation can be extracted it can be: The target type should be a DOI or OPENAIRE The target PID should not be null.
     *
     * @param relation the relation
     * @return if the relation can be extracted
     */
    private boolean canBeRelationExtracted(final DLIRelation relation) {
        // return (relation.getTargetPID() != null) && (relation.getTargetPID().getType() != null) &&
        // isPidTypeResolvable(relation.getTargetPID().getType().toLowerCase().trim());
        return true;
    }

    // private boolean isPidTypeResolvable(final String pidType) {
    // for(String s: resolvablePidType) {
    // if (s.equals(pidType))
    // return true;
    // }
    // return false;
    // }

    private void normalizeDOI(final DLIObject input) {
        if (input.getPidType() != null && input.getPidType().toLowerCase().trim().equals("doi")) {
            String pid = input.getPid();
            if (pid == null)
                return;
            pid = pid.replace("http://dx.doi.org/", "");
            input.setPid(pid);
        }
    }

    /**
     * Fix the record checking the provenance if it's null they will be setted with the right value.
     *
     * @param input the input
     */
    private void fixRecordProvenance(final DLIObject input) {
        if (input.getDatasourceProvenance() == null || input.getDatasourceProvenance().size() == 0) {
            DLICompletionStatus status = DLICompletionStatus.incomplete;
            // CHECK IF THE STATUS IS COMPLETE OR INCOMPLETE
            if (input.getType() != null || input.getTitles() != null && input.getTitles().length > 0
                    || input.getAuthors() != null && input.getAuthors().length > 0) {
                status = DLICompletionStatus.complete;
            }
            DLIProvenance prov = new DLIProvenance(namespace, DLIProvisionMode.collected.toString(), status.toString(), null, null, true);
            input.setDatasourceProvenance(Lists.newArrayList(prov));
        }
    }
}
