package eu.dnetlib.msro.dli.objectstore;

import com.google.gson.Gson;
import com.mongodb.DB;
import com.mongodb.gridfs.GridFS;
import com.mongodb.gridfs.GridFSDBFile;
import com.mongodb.gridfs.GridFSInputFile;
import eu.dnetlib.msro.workflows.dli.model.DLIObject;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

/**
 * Created by Sandro La Bruzzo on 11/26/15.
 */
public class DLIObjectStore {

    private static final Log log = LogFactory.getLog(DLIObjectStore.class);
    private static final String COLLECTION_NAME = "store";

    private int totalItem = 0;

    private DB db;

    private GridFS collection;


    public void initialize() {
        collection = new GridFS(db, "store");
    }


    public boolean deleteStore() {
        try {
            db.getCollection(COLLECTION_NAME).drop();
            return true;
        } catch (Throwable e) {
            log.error("An error occur on deleting a store", e);
            return false;
        }
    }

    public int feedStore(final Iterable<DLIObject> data) {
        if (collection == null) {
            initialize();
        }

        totalItem = 0;
        int insertedRecords = 0;

        try {
            for (DLIObject record : data) {
                String identifier = record.getIdentifier();
                Gson g = new Gson();
                GridFSInputFile inputFile = collection.createFile(createCompressRecord(g.toJson(record)));
                inputFile.setFilename(identifier);
                inputFile.save();
                totalItem++;
            }
        } catch (Throwable e) {
            log.error("Error on inserting record", e);
        }
        return insertedRecords;
    }


    private InputStream createCompressRecord(final String record) throws IOException {
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        ZipOutputStream zos = new ZipOutputStream(os);
        ZipEntry entry = new ZipEntry("record");
        zos.putNextEntry(entry);
        zos.write(record.getBytes());
        zos.closeEntry();
        zos.flush();
        zos.close();
        return new ByteArrayInputStream(os.toByteArray());
    }


    private InputStream decompressRecord(final InputStream record) throws IOException {
        ZipInputStream zis = new ZipInputStream(record);
        final ZipEntry entry = zis.getNextEntry();
        if ("record".equals(entry.getName())) {
            return zis;

        }
        return null;
    }

    public DB getDb() {
        return db;
    }

    @Required
    public void setDb(DB db) {
        this.db = db;
    }

    public InputStream findRecord(String dli_id) throws IOException {
        if (collection == null)
            initialize();
        if (dli_id == null) return null;
        final GridFSDBFile gridFSDBFile = collection.findOne(dli_id);
        if (gridFSDBFile != null) {
            return decompressRecord(gridFSDBFile.getInputStream());
        }
        return null;
    }

    public int getTotalItem() {
        return totalItem;
    }
}
