package eu.dnetlib.msro.workflows.blacklist;

import java.io.StringReader;
import java.util.List;
import java.util.Set;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import com.google.common.collect.Sets;
import com.google.gson.Gson;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;
import eu.dnetlib.data.hadoop.rmi.HadoopService;
import eu.dnetlib.data.hadoop.rmi.hbase.Column;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.transform.xml.AbstractDNetOafXsltFunctions;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.openaire.blacklist.BlacklistManager;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * Created by alessia on 23/10/15.
 */
public class ResolveBlacklistJobNode extends SimpleJobNode {

	private static final Log log = LogFactory.getLog(ResolveBlacklistJobNode.class);

	@Autowired
    private BlacklistManager blacklistManager;
    @Autowired
    private UniqueServiceLocator serviceLocator;

    @Autowired
    private ResultSetClientFactory resultSetClientFactory;

    private String blacklistParamName;

	private String clusterName;
    private String tableName;


    @Override
    protected String execute(NodeToken token) throws Exception {
        final W3CEndpointReference epr = this.blacklistManager.getAcceptedBlacklistEntries();
	    final HadoopService hadoopService = serviceLocator.getService(HadoopService.class);
	    final Set<ResolvedBlacklistEntry> resolved = Sets.newHashSet();

	    final SAXReader r = new SAXReader();

	    for(String s : resultSetClientFactory.getClient(epr)) {
		    final Document doc = r.read(new StringReader(s));

		    if (log.isDebugEnabled()) {
			    log.debug(doc.asXML());
		    }

		    final ResolvedBlacklistEntry rbe = new ResolvedBlacklistEntry();

		    final String source = fullId(doc.valueOf("//FIELD[./@name='source_type']"), doc.valueOf("//FIELD[./@name='source']"));
		    rbe.setSource(getResolvedId(source, hadoopService.describeHBaseColumn(getClusterName(), getTableName(), source).getColumns()));

		    final String target = fullId(doc.valueOf("//FIELD[./@name='target_type']"), doc.valueOf("//FIELD[./@name='target']"));
		    rbe.setTarget(getResolvedId(target, hadoopService.describeHBaseColumn(getClusterName(), getTableName(), target).getColumns()));

		    final String relType = doc.valueOf("//FIELD[./@name='relationship']");
		    rbe.setRelType(relType);

		    resolved.add(rbe);
		    resolved.add(new ResolvedBlacklistEntry(source, target, relType));
	    }

	    if (resolved.isEmpty()) {
		    log.info("blacklist is empty, nothing to do");
		    return "done";
	    }

	    log.info("blacklist size: " + resolved.size());

	    token.getEnv().setAttribute(getBlacklistParamName(), new Gson().toJson(resolved));

        return Arc.DEFAULT_ARC;
    }

	private String fullId(final String type, final String id) {
		final String fullId = AbstractDNetOafXsltFunctions.oafSimpleId(type, id);

		return OafRowKeyDecoder.decode(fullId).getKey();
	}

	private String getResolvedId(String originalId, List<Column> columns) {
		if (StringUtils.isBlank(originalId)) {
			throw new IllegalArgumentException("cannot resolve an empty id");
		}

	    for (Column c : columns) {
            if (c.getFamily().contains("isMergedIn")) {

	            for (String s : c.getQualifier()) {
		            if (StringUtils.isNotBlank(s)) return s;
	            }
            }
        }
        return originalId;
    }

    public String getBlacklistParamName() {
        return blacklistParamName;
    }

    public void setBlacklistParamName(String blacklistParamName) {
        this.blacklistParamName = blacklistParamName;
    }

	public String getClusterName() {
		return clusterName;
	}

	public void setClusterName(final String clusterName) {
		this.clusterName = clusterName;
	}

	public String getTableName() {
		return tableName;
	}

	public void setTableName(final String tableName) {
		this.tableName = tableName;
	}

}
