package eu.dnetlib.data.information.oai.publisher.conf;

import java.io.IOException;
import java.util.Collection;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.PublisherField;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.information.oai.publisher.info.SetInfo;

/**
 * Instances of this class reads the OAI configuration from a string, which is the configuration profile passed in as a string.
 * 
 * @author alessia
 * 
 */
public class OAIConfigurationStringReader implements OAIConfigurationReader {

	private static final Log log = LogFactory.getLog(OAIConfigurationStringReader.class); // NOPMD by marko on 11/24/08 5:02 PM

	private OAIConfiguration oaiConfiguration;
	@Resource
	private OAIConfigurationParser configurationParser;

	public OAIConfigurationStringReader() {
		oaiConfiguration = null;
	}

	public OAIConfigurationStringReader(final String profile) {
		this.readConfiguration(profile);
	}

	public void readConfiguration(final String profile) {
		log.debug(profile);
		try {
			oaiConfiguration = configurationParser.getConfiguration(profile);
		} catch (IOException e) {
			throw new OaiPublisherRuntimeException("Could not read OAI configuration profile", e);
		}
	}

	public boolean isConfigurationLoaded() {
		return this.oaiConfiguration != null;
	}

	@Override
	public List<SetInfo> getSets() {
		if (isConfigurationLoaded()) return Lists.newArrayList(this.oaiConfiguration.getSetsMap().values());
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public List<SetInfo> getSets(final boolean onlyEnabled) {
		if (isConfigurationLoaded()) return Lists.newArrayList(Iterables.filter(this.oaiConfiguration.getSetsMap().values(), new Predicate<SetInfo>() {

			/**
			 * {@inheritDoc}
			 * 
			 * @see com.google.common.base.Predicate#apply(java.lang.Object)
			 */
			@Override
			public boolean apply(final SetInfo setInfo) {
				if (onlyEnabled) return setInfo.isEnabled();
				else return true;
			}

		}));
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public List<String> getSetSpecs() {
		if (isConfigurationLoaded()) return Lists.newArrayList(this.oaiConfiguration.getSetsMap().keySet());
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public List<String> getSetSpecs(final boolean onlyEnabled) {
		if (isConfigurationLoaded()) {
			if (!onlyEnabled) return this.getSetSpecs();
			List<SetInfo> enabled = this.getSets(true);
			return Lists.newArrayList(Iterables.transform(enabled, new Function<SetInfo, String>() {

				@Override
				public String apply(final SetInfo setInfo) {
					return setInfo.getSetSpec();
				}

			}));
		} else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public List<MDFInfo> getSourceMetadataFormats() {
		if (isConfigurationLoaded()) return Lists.newArrayList(this.oaiConfiguration.getSourcesMDF());
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public SetInfo getSetInfo(final String setSpec) {
		if (isConfigurationLoaded()) return this.oaiConfiguration.getSetsMap().get(setSpec);
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public List<MDFInfo> getMetadataFormatInfo() {
		if (isConfigurationLoaded()) return Lists.newArrayList(this.oaiConfiguration.getMdFormatsMap().values());
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public List<MDFInfo> getMetadataFormatInfo(final boolean onlyEnabled) {
		if (isConfigurationLoaded()) {
			if (!onlyEnabled) return this.getMetadataFormatInfo();
			else return Lists.newArrayList(Iterables.filter(this.oaiConfiguration.getMdFormatsMap().values(), new Predicate<MDFInfo>() {

				/**
				 * {@inheritDoc}
				 * 
				 * @see com.google.common.base.Predicate#apply(java.lang.Object)
				 */
				@Override
				public boolean apply(final MDFInfo mdfInfo) {
					if (onlyEnabled) return mdfInfo.isEnabled();
					else return true;
				}

			}));

		} else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public MDFInfo getMetadataFormatInfo(final String mdPrefix) {
		if (isConfigurationLoaded()) return this.oaiConfiguration.getMdFormatsMap().get(mdPrefix);
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public List<PublisherField> getFields() {
		if (isConfigurationLoaded()) return this.oaiConfiguration.getFields();
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public List<PublisherField> getFields(final String format, final String interpretation, final String layout) {
		if (isConfigurationLoaded()) return Lists.newArrayList(this.oaiConfiguration.getFieldsFor(format, layout, interpretation));
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public List<String> getFieldNames() {
		if (isConfigurationLoaded()) return this.oaiConfiguration.getFieldNames();
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public List<MDFInfo> getFormatsServedBy(final String sourceFormatName, final String sourceFormatLayout, final String sourceFormatInterpretation) {
		if (isConfigurationLoaded()) {
			Collection<MDFInfo> unfiltered = this.oaiConfiguration.getMdFormatsMap().values();
			return Lists.newArrayList(Collections2.filter(unfiltered, new Predicate<MDFInfo>() {

				@Override
				public boolean apply(final MDFInfo mdf) {
					return (mdf.getSourceFormatName() == sourceFormatName) && (mdf.getSourceFormatLayout() == sourceFormatLayout)
							&& (mdf.getSourceFormatInterpretation() == sourceFormatInterpretation);
				}

			}));

		} else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public String getIdScheme() {
		if (isConfigurationLoaded()) return this.oaiConfiguration.getIdScheme();
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	@Override
	public String getIdNamespace() {
		if (isConfigurationLoaded()) return this.oaiConfiguration.getIdNamespace();
		else throw new OaiPublisherRuntimeException("Configuration is not loaded");
	}

	public OAIConfiguration getOaiConfiguration() {
		return oaiConfiguration;
	}

	public void setOaiConfiguration(final OAIConfiguration oaiConfiguration) {
		this.oaiConfiguration = oaiConfiguration;
	}

	public OAIConfigurationParser getConfigurationParser() {
		return configurationParser;
	}

	public void setConfigurationParser(final OAIConfigurationParser configurationParser) {
		this.configurationParser = configurationParser;
	}

}
