package eu.dnetlib.data.information.oai.publisher.conf;

import java.io.IOException;
import java.io.StringReader;
import java.util.List;
import java.util.Map;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.PublisherField;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.information.oai.publisher.info.SetInfo;

/**
 * Parses an XML document representing the OAI configuration profile and creates the corresponding OAIConfiguration object.
 * 
 * @author alessia
 * 
 */
public class OAIConfigurationParser {

	private static final Log log = LogFactory.getLog(OAIConfigurationParser.class); // NOPMD by marko on 11/24/08 5:02 PM

	private final ThreadLocal<XMLInputFactory> factory = new ThreadLocal<XMLInputFactory>() {

		@Override
		protected XMLInputFactory initialValue() {
			return XMLInputFactory.newInstance();
		}
	};

	public OAIConfiguration getConfiguration(final String configurationProfile) throws IOException {
		log.info("Parsing OAI configuration profile");
		log.debug(configurationProfile);
		OAIConfiguration config = new OAIConfiguration();
		Map<String, SetInfo> setsMap = Maps.newHashMap();
		Map<String, MDFInfo> mdFormatsMap = Maps.newHashMap();
		List<String> indexNames = Lists.newArrayList();
		List<PublisherField> fields = Lists.newArrayList();
		try {
			final XMLStreamReader parser = factory.get().createXMLStreamReader(new StreamSource(new StringReader(configurationProfile)));
			log.debug("Configuration profile read by " + parser.getClass().getCanonicalName());

			while (parser.hasNext()) {
				int event = parser.next();
				if (event == XMLStreamConstants.START_ELEMENT) {
					final String localName = parser.getLocalName();
					if (localName.equals("OAISET")) {
						boolean inSet = true;
						SetInfo setInfo = new SetInfo();
						String enabled = parser.getAttributeValue(null, "enabled");
						setInfo.setEnabled(Boolean.parseBoolean(enabled));
						while (parser.hasNext() && inSet) {
							event = parser.next();
							if (event == XMLStreamConstants.START_ELEMENT) {
								String setElementName = parser.getLocalName();
								String setElementValue = parser.getElementText();
								this.handleSetInfo(setInfo, setElementName, setElementValue);
							}
							if ((event == XMLStreamConstants.END_ELEMENT) && parser.getLocalName().equals("OAISET")) {
								inSet = false;
								setsMap.put(setInfo.getSetSpec(), setInfo);
							}
						}
					} else {
						if (localName.equals("METADATAFORMAT")) {
							boolean inMetadata = true;
							MDFInfo mdfInfo = new MDFInfo();
							String exportable = parser.getAttributeValue(null, "exportable");
							mdfInfo.setEnabled(Boolean.parseBoolean(exportable));
							String mdPrefix = parser.getAttributeValue(null, "metadataPrefix");
							mdfInfo.setPrefix(mdPrefix);
							while (parser.hasNext() && inMetadata) {
								event = parser.next();
								if (event == XMLStreamConstants.START_ELEMENT) {
									String mdfElementName = parser.getLocalName();
									if (mdfElementName.equals("SOURCE_METADATA_FORMAT")) {
										this.handleSourceMDF(mdfInfo, parser);
										config.getSourcesMDF().add(mdfInfo);
									} else {
										String mdfElementValue = parser.getElementText();
										this.handleMDFInfo(mdfInfo, mdfElementName, mdfElementValue);
									}
								}
								if ((event == XMLStreamConstants.END_ELEMENT) && parser.getLocalName().equals("METADATAFORMAT")) {
									inMetadata = false;
									mdFormatsMap.put(mdPrefix, mdfInfo);
								}
							}
						} else {
							// INDICES
							if (localName.equals("INDEX")) {
								boolean inIndex = true;
								PublisherField publisherField = new PublisherField();
								String indexName = parser.getAttributeValue(null, "name");
								String repeatable = parser.getAttributeValue(null, "repeatable");
								boolean isRepeatable = Boolean.valueOf(repeatable);
								indexNames.add(indexName);
								publisherField.setFieldName(indexName);
								publisherField.setRepeatable(isRepeatable);
								// now let's set the SOURCES
								Multimap<String, String> fieldSources = ArrayListMultimap.create();
								while (parser.hasNext() && inIndex) {
									event = parser.next();
									if (event == XMLStreamConstants.START_ELEMENT) {
										String currentElementName = parser.getLocalName();
										this.handleIndex(fieldSources, indexName, parser, currentElementName);
									}
									if ((event == XMLStreamConstants.END_ELEMENT) && parser.getLocalName().equals("INDEX")) {
										inIndex = false;
									}
								}
								publisherField.setSources(fieldSources);
								fields.add(publisherField);

							}
						}
					}
				}
			}
			config.setFields(fields);
			config.setFieldNames(indexNames);
			config.setMdFormatsMap(mdFormatsMap);
			config.setSetsMap(setsMap);
			return config;
		} catch (final XMLStreamException e) {
			throw new OaiPublisherRuntimeException(e);
		}

	}

	/**
	 * Sets information about the indices.
	 * 
	 * @param fieldSources
	 * @param indexName
	 * @param parser
	 * @param currentLocalName
	 */
	private void handleIndex(final Multimap<String, String> fieldSources, final String indexName, final XMLStreamReader parser, final String currentLocalName) {
		if (currentLocalName.equals("SOURCE")) {
			MDFInfo indexSource = new MDFInfo();
			this.handleSourceMDF(indexSource, parser);
			String key = indexSource.getSourceFormatName() + "-" + indexSource.getSourceFormatLayout() + "-" + indexSource.getSourceFormatInterpretation();
			fieldSources.put(key, parser.getAttributeValue(null, "path"));
		} else {
			log.warn("I do not know how to handle INDEX element with name " + currentLocalName);
		}
	}

	/**
	 * Sets information about the source metadata format in the given instance of MDFInfo.
	 * 
	 * @param mdfInfo
	 * @param parser
	 */
	private void handleSourceMDF(final MDFInfo mdfInfo, final XMLStreamReader parser) {
		String interpretation = parser.getAttributeValue(null, "interpretation");
		String layout = parser.getAttributeValue(null, "layout");
		String name = parser.getAttributeValue(null, "name");
		mdfInfo.setSourceFormatInterpretation(interpretation);
		mdfInfo.setSourceFormatLayout(layout);
		mdfInfo.setSourceFormatName(name);
	}

	/**
	 * Sets information in the MDFInfo instance based on the element name.
	 * 
	 * @param mdfInfo
	 * @param mdfElementName
	 * @param mdfElementValue
	 */
	private void handleMDFInfo(final MDFInfo mdfInfo, final String elementName, final String elementValue) {
		if (elementName.equals("NAMESPACE")) {
			mdfInfo.setNamespace(elementValue);
		} else {
			if (elementName.equals("SCHEMA")) {
				mdfInfo.setSchema(elementValue);
			} else {
				if (elementName.equals("TRANSFORMATION_RULE")) {
					mdfInfo.setTransformationRuleID(elementValue);
				} else {
					if (elementName.equals("BASE_QUERY")) {
						mdfInfo.setBaseQuery(elementValue);
					} else {
						log.warn("I do not know how to handle Metadata Format element with name " + elementName + " and value " + elementValue);
					}
				}
			}
		}

	}

	/**
	 * Sets information in the SetInfo instance based on the element name.
	 * 
	 * @param setInfo
	 * @param elementName
	 * @param elementValue
	 */
	private void handleSetInfo(final SetInfo setInfo, final String elementName, final String elementValue) {
		if (elementName.equals("spec")) {
			setInfo.setSetSpec(elementValue);
		} else {
			if (elementName.equals("name")) {
				setInfo.setSetName(elementValue);
			} else {
				if (elementName.equals("description")) {
					setInfo.setSetDescription(elementValue);
				} else {
					if (elementName.equals("query")) {
						setInfo.setQuery(elementValue);
					} else {
						log.warn("I do not know how to handle Set element with name " + elementName + " and value " + elementValue);
					}
				}
			}
		}
	}
}
