package eu.dnetlib.data.oai.store.sets;

import java.text.Normalizer;
import java.util.List;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.collect.Lists;
import com.mongodb.BasicDBObject;
import com.mongodb.BasicDBObjectBuilder;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;
import com.mongodb.Mongo;

import eu.dnetlib.data.information.oai.publisher.info.SetInfo;
import eu.dnetlib.data.information.oai.sets.SetCollection;

public class MongoSetCollection implements SetCollection {

	public static String DEFAULT_SET = "OTHER";
	@Autowired
	private Mongo publisherMongoServer;
	private String setCollection = "sets";
	private String setCountCollection = "setsCount";

	public void ensureIndexes(final String dbName) {
		this.ensureIndexesOnSets(dbName);
		this.ensureIndexesOnCount(dbName);
	}

	@Override
	public List<SetInfo> getAllSets(final boolean enabledOnly, final String dbName) {
		DBCursor cursor = null;
		if (!enabledOnly) {
			cursor = this.getSetsCollection(dbName).find();
		} else {
			DBObject where = new BasicDBObject("enabled", true);
			cursor = this.getSetsCollection(dbName).find(where);
		}
		List<SetInfo> res = Lists.newArrayList();
		while (cursor.hasNext()) {
			DBObject obj = cursor.next();
			res.add(this.getSetFromDBObject(obj));
		}
		return res;
	}

	@Override
	public boolean containSet(final String set, final String dbName) {
		DBObject query = new BasicDBObject("spec", set);
		return this.getSetsCollection(dbName).find(query).limit(1).size() != 0;
	}

	@Override
	public boolean containEnabledSet(final String set, final String publisherDBName) {
		DBObject query = new BasicDBObject("spec", set).append("enabled", true);
		return this.getSetsCollection(publisherDBName).find(query).limit(1).size() != 0;
	}

	@Override
	public String getSetQuery(final String set, final String dbName) {
		DBObject query = new BasicDBObject("spec", set);
		DBObject returnField = new BasicDBObject("query", 1);
		DBObject obj = this.getSetsCollection(dbName).findOne(query, returnField);
		return (String) obj.get("query");
	}

	@Override
	public int count(final String setSpec, final String mdPrefix, final String dbName) {
		DBObject query = BasicDBObjectBuilder.start("spec", setSpec).add("mdPrefix", mdPrefix).get();
		DBObject returnField = new BasicDBObject("count", 1);
		DBObject obj = this.getSetsCountCollection(dbName).findOne(query, returnField);
		return (Integer) obj.get("count");
	}

	public void updateCounts(final String setSpec, final String mdPrefix, final int count, final String dbName) {
		DBObject setCount = BasicDBObjectBuilder.start("spec", setSpec).add("mdPrefix", mdPrefix).get();
		this.getSetsCountCollection(dbName).update(setCount, new BasicDBObject("$set", new BasicDBObject("count", count)), true, false);
	}

	public void upsertSet(final SetInfo setInfo, final boolean fromConfiguration, final String dbName) {
		DBObject obj = this.getObjectFromSet(setInfo);
		obj.put("fromConfiguration", fromConfiguration);
		this.getSetsCollection(dbName).update(new BasicDBObject("spec", setInfo.getSetSpec()), obj, true, false);
	}

	public String normalizeSetSpec(final String setName) {
		String s = StringEscapeUtils.unescapeXml(setName);
		s = Normalizer.normalize(s, Normalizer.Form.NFD);
		// replace spaces with underscores
		s = s.replaceAll(" ", "_");
		// remove tilde, dots... over letters
		s = s.replaceAll("[\\p{InCombiningDiacriticalMarks}&&[^-_]]", "");
		// change punctuation into an underscore
		s = s.replaceAll("[\\p{Punct}&&[^-_]]", "_");
		// remove all non-word characters
		s = s.replaceAll("[\\W&&[^-_]]", "");
		// Avoiding set '___' generated when we have "strange" set names such as those in cyrillic/ukrain
		// strips _ from the beginning and the end
		String stripped = StringUtils.strip(s, "_ ");
		if (StringUtils.isBlank(stripped)) {
			stripped = DEFAULT_SET;
		}
		return stripped;
	}

	public List<SetInfo> getConfiguredSets(final String dbName) {
		DBObject query = new BasicDBObject("fromConfiguration", true);
		return this.findSets(query, dbName);
	}

	public List<SetInfo> getSetsFromData(final String dbName) {
		DBObject query = new BasicDBObject("fromConfiguration", false);
		return this.findSets(query, dbName);
	}

	public void dropOAISets(final String dbName) {
		this.getSetsCountCollection(dbName).drop();
		this.getSetsCollection(dbName).drop();
	}

	public void dropSet(final String dbName, final String setSpec) {
		DBObject query = BasicDBObjectBuilder.start("spec", setSpec).get();
		this.getSetsCollection(dbName).remove(query);
		this.getSetsCountCollection(dbName).remove(query);
	}

	public void dropConfigurationSets(final String dbName) {
		this.getSetsCollection(dbName).remove(new BasicDBObject("fromConfiguration", true));
	}

	protected List<SetInfo> findSets(final DBObject query, final String dbName) {
		DBCursor cursor = this.getSetsCollection(dbName).find(query);
		List<SetInfo> res = Lists.newArrayList();
		while (cursor.hasNext()) {
			DBObject obj = cursor.next();
			res.add(this.getSetFromDBObject(obj));
		}
		return res;
	}

	private SetInfo getSetFromDBObject(final DBObject obj) {
		SetInfo setInfo = new SetInfo();
		setInfo.setEnabled((Boolean) obj.get("enabled"));
		setInfo.setQuery((String) obj.get("query"));
		setInfo.setSetDescription((String) obj.get("description"));
		setInfo.setSetName((String) obj.get("name"));
		setInfo.setSetSpec((String) obj.get("spec"));
		return setInfo;
	}

	private DBObject getObjectFromSet(final SetInfo s) {
		DBObject obj = BasicDBObjectBuilder.start("spec", s.getSetSpec()).add("name", s.getSetName()).add("description", s.getSetDescription())
				.add("query", s.getQuery()).add("enabled", s.isEnabled()).get();
		return obj;
	}

	private void ensureIndexesOnSets(final String dbName) {
		DBObject onBackground = new BasicDBObject("background", true);
		this.getSetsCollection(dbName).ensureIndex(new BasicDBObject("spec", 1), onBackground);
		this.getSetsCollection(dbName).ensureIndex(new BasicDBObject("fromConfiguration", 1), onBackground);
	}

	private void ensureIndexesOnCount(final String dbName) {
		DBObject index = BasicDBObjectBuilder.start("spec", 1).add("mdPrefix", 1).get();
		this.getSetsCountCollection(dbName).ensureIndex(index, new BasicDBObject("background", true));
	}

	public DBCollection getSetsCollection(final String dbName) {
		return this.getCollection(this.setCollection, dbName);
	}

	public DBCollection getSetsCountCollection(final String dbName) {
		return this.getCollection(this.setCountCollection, dbName);
	}

	private DBCollection getCollection(final String collectionName, final String dbName) {
		return publisherMongoServer.getDB(dbName).getCollection(collectionName);
	}

	public String getSetCollection() {
		return setCollection;
	}

	public void setSetCollection(final String setCollection) {
		this.setCollection = setCollection;
	}

	public String getSetCountCollection() {
		return setCountCollection;
	}

	public void setSetCountCollection(final String setCountCollection) {
		this.setCountCollection = setCountCollection;
	}

	public Mongo getPublisherMongoServer() {
		return publisherMongoServer;
	}

	public void setPublisherMongoServer(final Mongo publisherMongoServer) {
		this.publisherMongoServer = publisherMongoServer;
	}

}
