package eu.dnetlib.data.oai.store;

import java.util.List;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherException;

public interface PublisherStoreDAO<X extends PublisherStore<T>, T extends Cursor> {

	/**
	 * Lists all PublisherStore.
	 * 
	 * @param dbName
	 *            name of the target database
	 * 
	 * @return a List of PublisherStore instances.
	 */
	List<X> listPublisherStores(final String dbName);

	/**
	 * Gets the store with the given identifier.
	 * 
	 * @param dbName
	 *            name of the target database
	 * 
	 * @param storeId
	 *            identifier of the store to retrieve
	 * @return a PublisherStore instance or null if there is no store with the given id.
	 */
	X getStore(final String storeId, final String dbName);

	/**
	 * Gets the store with the given properties.
	 * 
	 * @param mdfName
	 *            name of the metadata format
	 * @param mdfInterpretation
	 *            name of the metadata interpretation
	 * @param mdfLayout
	 *            name of the metadata layout
	 * @param dbName
	 *            name of the target database
	 * @return a PublisherStore instance or null if there is no store with the given properties.
	 */
	X getStore(final String mdfName, final String mdfInterpretation, final String mdfLayout, final String dbName);

	/**
	 * Gets the store to be used as source to deliver records with the given metadata prefix.
	 * 
	 * @param targetMetadataPrefix
	 *            prefix of the metadata format deliverable through this store
	 * @param dbName
	 *            name of the target database
	 * @return a PublisherStore instance or null if there is no store serving the given metadata prefix.
	 */
	X getStoreFor(final String targetMetadataPrefix, final String dbName);

	/**
	 * Create a PublisherStore with the given properties.
	 * 
	 * @param mdfName
	 *            name of the metadata format
	 * @param mdfInterpretation
	 *            name of the metadata interpretation
	 * @param mdfLayout
	 *            name of the metadata layout
	 * @param dbName
	 *            name of the target database
	 * @return a PublisherStore instance whose identifier is automatically generated.
	 * @throws OaiPublisherException
	 *             if there is already another PublisherStore with the given metadata format name, layout and interpretation.
	 */
	X createStore(final String mdfName, final String mdfInterpretation, final String mdfLayout, final String dbName) throws OaiPublisherException;

	/**
	 * Deletes the store with the given identifier.
	 * 
	 * @param storeId
	 *            id of the store to delete
	 * @param dbName
	 *            name of the target database
	 * @return true if the store was deleted successfully, false otherwise (e.g., a store with the given id does not exist).
	 */
	boolean deleteStore(final String storeId, final String dbName);

	/**
	 * Deletes from the store with the given identifier the records belonging to the given set.
	 * 
	 * @param storeId
	 *            id of the store to delete
	 * @param dbName
	 *            name of the target database
	 * @param set
	 *            name of the set
	 * @return true if the records were deleted successfully, false otherwise (e.g., a store with the given id does not exist).
	 */
	boolean deleteFromStore(final String storeId, final String dbName, final String set);

	/**
	 * Deletes from the store with the given identifier the records belonging to the given set.
	 * 
	 * @param mdfName
	 *            name of the metadata format
	 * @param mdfInterpretation
	 *            name of the metadata interpretation
	 * @param mdfLayout
	 *            name of the metadata layout
	 * @param dbName
	 *            name of the target database
	 * @param set
	 *            name of the set
	 * @return true if the records were deleted successfully, false otherwise (e.g., a store with the given id does not exist).
	 */
	boolean deleteFromStore(final String mdfName, final String mdfInterpretation, final String mdfLayout, final String dbName, final String set);

	/**
	 * Deletes the store with the given properties.
	 * 
	 * @param mdfName
	 *            name of the metadata format
	 * @param mdfInterpretation
	 *            name of the metadata interpretation
	 * @param mdfLayout
	 *            name of the metadata layout
	 * @param dbName
	 *            name of the target database
	 * @return true if the store was deleted successfully, false otherwise (e.g., a store with the given properties does not exist).
	 */
	boolean deleteStore(final String mdfName, final String mdfInterpretation, final String mdfLayout, final String dbName);

}
