package eu.dnetlib.msro.workflows.nodes.objectStore;

import java.util.List;
import java.util.Set;

import javax.annotation.Resource;

import org.springframework.beans.factory.annotation.Required;

import com.google.common.collect.Sets;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;

/**
 * The Class RetrieveMdStoreId is a job node used to retrieve the correct MDStore from which extract the url of the file to download.
 * metadata format and interpretation are injected as properties
 */
public class RetrieveMdStoreId extends SimpleJobNode {

	/** The metadata format. */
	private String metadataFormat;

	/** The interpretation. */
	private String interpretation;

	/** The provider id. */
	private String providerId;

	/** The lookup locator. */
	@Resource(name = "lookupLocator")
	private ServiceLocator<ISLookUpService> lookupLocator;

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.msro.workflows.nodes.SimpleJobNode#execute(com.googlecode.sarasvati.NodeToken)
	 */
	@Override
	protected String execute(final NodeToken token) throws Exception {

		String query = "for $x in collection('/db/DRIVER/WorkflowDSResources/WorkflowDSResourceType') "
				+ "where ($x//PARAM/@name=\"mdId\" and $x//PARAM/@name=\"providerId\" and " + "$x//PARAM[./@name=\"providerId\"]/text()=\"%s\")"
				+ "return distinct-values($x//PARAM[./@name=\"mdId\"])";

		List<String> result = lookupLocator.getService().quickSearchProfile(String.format(query, providerId));
		if (result.size() > 0) {

		} else throw new RuntimeException("there is no mdStore Associated to the provider " + token.getEnv().getAttribute(getProviderId()));

		Set<String> metadataIds = Sets.newHashSet(result);
		String id = getRightMetadataId(metadataIds);
		token.getEnv().setAttribute("mdId", id);
		token.getEnv().setAttribute("mdFormat", getMetadataFormat());

		return Arc.DEFAULT_ARC;
	}

	/**
	 * Gets the right metadata id whith the format metadataFormat and interpretation interpretation
	 * 
	 * @return the right metadata id
	 * @throws ISLookUpException
	 */
	private String getRightMetadataId(final Iterable<String> ids) throws ISLookUpException {
		String query = "concat(//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value=\"%s\"]//METADATA_FORMAT/text(), \"::<<>>::\""
				+ ",//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value=\"%s\"]//METADATA_FORMAT_INTERPRETATION/text())";

		for (String id : ids) {

			List<String> results = lookupLocator.getService().quickSearchProfile(String.format(query, id, id));
			if (results.size() > 0) {
				String[] values = results.get(0).split("::<<>>::");
				if (metadataFormat.equals(values[0]) && interpretation.equals(values[1])) return id;
			}
		}
		return null;
	}

	/**
	 * Gets the interpretation.
	 * 
	 * @return the interpretation
	 */
	public String getInterpretation() {
		return interpretation;
	}

	/**
	 * Sets the interpretation.
	 * 
	 * @param interpretation
	 *            the interpretation to set
	 */
	@Required
	public void setInterpretation(final String interpretation) {
		this.interpretation = interpretation;
	}

	/**
	 * Gets the metadata format.
	 * 
	 * @return the metadataFormat
	 */
	public String getMetadataFormat() {
		return metadataFormat;
	}

	/**
	 * Sets the metadata format.
	 * 
	 * @param metadataFormat
	 *            the metadataFormat to set
	 */
	@Required
	public void setMetadataFormat(final String metadataFormat) {
		this.metadataFormat = metadataFormat;
	}

	/**
	 * Gets the provider id.
	 * 
	 * @return the providerId
	 */
	public String getProviderId() {
		return providerId;
	}

	/**
	 * Sets the provider id.
	 * 
	 * @param providerId
	 *            the providerId to set
	 */
	public void setProviderId(final String providerId) {
		this.providerId = providerId;
	}

}
