package eu.dnetlib.functionality.modular.ui.repositories;

import java.io.StringReader;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;

import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.ClassPathResource;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import eu.dnetlib.data.collector.rmi.CollectorService;
import eu.dnetlib.data.collector.rmi.ProtocolDescriptor;
import eu.dnetlib.datasource.common.utils.DefaultDatasourceUpdater;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.functionality.modular.ui.repositories.objects.RepoEntry;
import eu.dnetlib.functionality.modular.ui.repositories.objects.RepoInterfaceEntry;
import eu.dnetlib.functionality.modular.ui.repositories.objects.RepoMapEntry;
import eu.dnetlib.functionality.modular.ui.repositories.objects.RepoMetaWfEntry;
import eu.dnetlib.functionality.modular.ui.repositories.objects.SimpleParamEntry;
import eu.dnetlib.functionality.modular.ui.repositories.objects.SimpleRepoInterfaceEntry;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants.WorkflowStatus;

public class RepoUIUtils {

	@Resource
	private UniqueServiceLocator serviceLocator;

	private ClassPathResource browseRepoApisQueryTmpl = new ClassPathResource(
			"/eu/dnetlib/functionality/modular/ui/repositories/templates/browseRepoApis.xquery.st");

	private ClassPathResource getRepoApiQueryTmpl = new ClassPathResource("/eu/dnetlib/functionality/modular/ui/repositories/templates/getRepoApi.xquery.st");

	private ClassPathResource findRepoApisQueryTmpl = new ClassPathResource(
			"/eu/dnetlib/functionality/modular/ui/repositories/templates/findRepoApis.xquery.st");

	private ClassPathResource findReposQueryTmpl = new ClassPathResource("/eu/dnetlib/functionality/modular/ui/repositories/templates/findRepos.xquery.st");

	private ClassPathResource findReposMapQuery = new ClassPathResource("/eu/dnetlib/functionality/modular/ui/repositories/templates/findReposMap.xquery");

	private static final Log log = LogFactory.getLog(RepoUIUtils.class);

	private String browseFieldsJson;

	private Map<String, List<String>> parametersMap = Maps.newHashMap();

	public List<SimpleParamEntry> browseRepoField(final String xpath, final String vocabulary) throws Exception {
		final StringTemplate st = new StringTemplate(IOUtils.toString(browseRepoApisQueryTmpl.getInputStream()));
		st.setAttribute("xpath", xpath);

		final Map<String, String> voc = getVocabulary(vocabulary);

		final List<SimpleParamEntry> list = Lists.newArrayList();
		for (String s : serviceLocator.getService(ISLookUpService.class).quickSearchProfile(st.toString())) {
			final String[] arr = s.split("@-@-@");
			final String id = arr[0].trim();
			final String count = arr[1].trim();
			list.add(new SimpleParamEntry(id, findLabel(id, voc), count));
		}

		return list;
	}

	public List<SimpleRepoInterfaceEntry> listApis(final String param, final String value, final String xpath) throws Exception {
		final StringTemplate st = new StringTemplate();
		if (param.equalsIgnoreCase("__search__")) {
			st.setTemplate(IOUtils.toString(findRepoApisQueryTmpl.getInputStream()));
			st.setAttribute("cond", "contains(../..//(*|@*)/lower-case(.), '" + StringEscapeUtils.escapeXml(value.toLowerCase()) + "')");
		} else {
			st.setTemplate(IOUtils.toString(findRepoApisQueryTmpl.getInputStream()));
			st.setAttribute("cond", xpath + "='" + StringEscapeUtils.escapeXml(value) + "'");
		}

		final String query = st.toString();

		final SAXReader reader = new SAXReader();

		final List<SimpleRepoInterfaceEntry> list = Lists.transform(serviceLocator.getService(ISLookUpService.class).quickSearchProfile(query),
				new Function<String, SimpleRepoInterfaceEntry>() {

					@Override
					public SimpleRepoInterfaceEntry apply(final String s) {
						final SimpleRepoInterfaceEntry iface = new SimpleRepoInterfaceEntry();
						try {
							final Document doc = reader.read(new StringReader(s));
							final String country = doc.valueOf("//REPO/@country");

							iface.setRepoId(doc.valueOf("//REPO/@id"));
							iface.setRepoCountry(StringUtils.isEmpty(country) ? "-" : country.toUpperCase());
							iface.setRepoName(doc.valueOf("//REPO/@name"));
							iface.setRepoPrefix(doc.valueOf("//REPO/@prefix"));

							final Node ifcNode = doc.selectSingleNode("//INTERFACE");

							iface.setId(ifcNode.valueOf("./@id"));
							iface.setActive(Boolean.valueOf(ifcNode.valueOf("./@active")));
							iface.setProtocol(ifcNode.valueOf("./ACCESS_PROTOCOL/text()"));

							final String overCompliance = ifcNode.valueOf("./INTERFACE_EXTRA_FIELD[@name='overriding_compliance']");
							if (StringUtils.isEmpty(overCompliance)) {
								iface.setCompliance(ifcNode.valueOf("@compliance"));
							} else {
								iface.setCompliance(overCompliance);
							}

							final String lastAggregationDate = ifcNode.valueOf("./INTERFACE_EXTRA_FIELD[@name='last_aggregation_date']");
							if (!StringUtils.isEmpty(lastAggregationDate)) {
								iface.setAggrDate(lastAggregationDate);
							} else {
								final String lastDownloadDate = ifcNode.valueOf("./INTERFACE_EXTRA_FIELD[@name='last_download_date']");
								if (!StringUtils.isEmpty(lastDownloadDate)) {
									iface.setAggrDate(lastDownloadDate);
								}
							}
							final String lastAggregationTotal = ifcNode.valueOf("./INTERFACE_EXTRA_FIELD[@name='last_aggregation_total']");
							if (StringUtils.isEmpty(lastAggregationTotal)) {
								final String lastDownloadTotal = ifcNode.valueOf("./INTERFACE_EXTRA_FIELD[@name='last_download_total']");
								if (StringUtils.isEmpty(lastDownloadTotal)) {
									iface.setAggrTotal(0);
								} else {
									iface.setAggrTotal(NumberUtils.toInt(lastDownloadTotal, 0));
								}
							} else {
								iface.setAggrTotal(NumberUtils.toInt(lastAggregationTotal, 0));
							}
						} catch (Exception e) {
							log.error(e);
						}
						return iface;
					}
				});
		return list;
	}

	public RepoInterfaceEntry getApi(final String repoId, final String ifaceId) throws Exception {
		final RepoInterfaceEntry ifc = new RepoInterfaceEntry();

		final StringTemplate st = new StringTemplate(IOUtils.toString(getRepoApiQueryTmpl.getInputStream()));
		st.setAttribute("dsId", repoId);
		st.setAttribute("ifaceId", ifaceId);

		final String query = st.toString();

		final SAXReader reader = new SAXReader();

		final String s = serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(query);

		final Document doc = reader.read(new StringReader(s));

		ifc.setId(doc.valueOf("/api/id"));
		ifc.setLabel(doc.valueOf("/api/label"));
		ifc.setRepoId(doc.valueOf("/api/repo/@id"));
		ifc.setRepoName(doc.valueOf("/api/repo/text()"));
		ifc.setRepoCountry(doc.valueOf("/api/repo/@country"));
		ifc.setRepoPrefix(doc.valueOf("/api/repo/@prefix"));

		final String protocol = doc.valueOf("/api/protocol");
		ifc.setProtocol(protocol);

		final Set<String> toVerifyParams = getParameterNamesForProtocol(ifc.getProtocol());
		for (Object o : doc.selectNodes("/api/commonParams/param")) {
			final Node n = (Node) o;
			ifc.getCommonParams().add(new SimpleParamEntry(n.valueOf("@name"), n.getText()));
		}

		log.debug("****** " + toVerifyParams);
		for (Object o : doc.selectNodes("/api/accessParams/param")) {
			final Node n = (Node) o;
			final String pname = n.valueOf("@name");
			if (toVerifyParams.contains(pname)) {
				ifc.getAccessParams().add(new SimpleParamEntry(pname, n.getText()));
				toVerifyParams.remove(pname);
			} else {
				log.warn("Invalid param [" + pname + "] for protocol " + protocol + " in repo " + repoId);
			}
		}
		for (String pname : toVerifyParams) {
			ifc.getAccessParams().add(new SimpleParamEntry(pname, ""));
			log.info("Adding missing param [" + pname + "] for protocol " + protocol + " in repo " + repoId);
		}

		for (Object o : doc.selectNodes("/api/extraFields/field")) {
			final Node n = (Node) o;
			final String name = n.valueOf("@name");
			final String value = n.getText();
			if (name.equalsIgnoreCase(DefaultDatasourceUpdater.OVERRIDING_COMPLIANCE_FIELD)) {
				for (SimpleParamEntry e : ifc.getCommonParams()) {
					if (e.getName().equals("compliance")) {
						// The original compliance (assigned by the validator) is stored in otherValue
						e.setOtherValue(e.getValue());
						e.setValue(value);
					}
				}
			} else if (name.equalsIgnoreCase("last_aggregation_date")) {
				ifc.setAggrDate(value);
			} else if (name.equalsIgnoreCase("last_aggregation_total")) {
				ifc.setAggrTotal(NumberUtils.toInt(value, 0));
			} else if (name.equalsIgnoreCase("last_aggregation_mdId")) {
				ifc.setAggrMdId(value);
			} else if (name.equalsIgnoreCase("last_collection_date")) {
				ifc.setCollDate(value);
			} else if (name.equalsIgnoreCase("last_collection_total")) {
				ifc.setCollTotal(NumberUtils.toInt(value, 0));
			} else if (name.equalsIgnoreCase("last_collection_mdId")) {
				ifc.setCollMdId(value);
			} else if (name.equalsIgnoreCase("last_download_date")) {
				ifc.setDownloadDate(value);
			} else if (name.equalsIgnoreCase("last_download_total")) {
				ifc.setDownloadTotal(NumberUtils.toInt(value, 0));
			} else if (name.equalsIgnoreCase("last_download_objId")) {
				ifc.setDownloadObjId(value);
			} else {
				ifc.getOtherParams().add(new SimpleParamEntry(name, value));
			}
		}

		for (Object o : doc.selectNodes("//metaWF")) {
			final Node n = (Node) o;

			final String id = n.valueOf("./id");
			final String name = n.valueOf("./name");
			final String status = n.valueOf("./status");
			final String repoByeWfId = n.valueOf("./destroyWorkflow");

			int progress = 0;
			try {
				switch (WorkflowStatus.valueOf(status)) {
				case MISSING:
					progress = 0;
					break;
				case ASSIGNED:
					progress = 25;
					break;
				case WAIT_SYS_SETTINGS:
					progress = 50;
					break;
				case WAIT_USER_SETTINGS:
					progress = 75;
					break;
				case EXECUTABLE:
					progress = 100;
					break;
				}
			} catch (Exception e) {
				progress = 0;
			}
			ifc.getMetaWFs().add(new RepoMetaWfEntry(id, name, status, repoByeWfId, progress));
		}

		return ifc;
	}

	public List<RepoEntry> listRepositories(final String type) throws Exception {
		final List<RepoEntry> list = Lists.newArrayList();

		final StringTemplate st = new StringTemplate(IOUtils.toString(findReposQueryTmpl.getInputStream()));
		st.setAttribute("type", type);

		for (String s : serviceLocator.getService(ISLookUpService.class).quickSearchProfile(st.toString())) {
			final RepoEntry r = new RepoEntry();
			final String[] arr = s.split("@=@");
			r.setId(arr[0].trim());
			r.setName(arr[1].trim());
			r.setValid("true".equals(arr[2].trim()));
			list.add(r);
		}

		Collections.sort(list);

		return list;
	}

	public List<RepoMapEntry> listRepositories_asMap() throws Exception {
		final SAXReader reader = new SAXReader();

		final String query = IOUtils.toString(findReposMapQuery.getInputStream());
		final List<RepoMapEntry> list = Lists.transform(serviceLocator.getService(ISLookUpService.class).quickSearchProfile(query),
				new Function<String, RepoMapEntry>() {

					@Override
					public RepoMapEntry apply(final String s) {
						final RepoMapEntry r = new RepoMapEntry();

						try {
							final Document doc = reader.read(new StringReader(s));
							r.setId(doc.valueOf("//dsId"));
							r.setName(doc.valueOf("//name"));
							r.setTot(Integer.parseInt(doc.valueOf("//size")));
							r.setLat(Float.parseFloat(doc.valueOf("//lat")));
							r.setLng(Float.parseFloat(doc.valueOf("//lng")));
						} catch (Exception e) {
							log.error(e);
						}
						return r;
					}
				});

		return list;
	}

	private String findLabel(final String id, final Map<String, String> vocabulary) {
		if (vocabulary.containsKey(id)) {
			return vocabulary.get(id);
		} else {
			return id;
		}
	}

	private Map<String, String> getVocabulary(final String vocabulary) throws ISLookUpException {
		final String query = "for $x in collection('/db/DRIVER/VocabularyDSResources/VocabularyDSResourceType')[.//VOCABULARY_NAME/@code = '" + vocabulary
				+ "']//TERM return concat($x/@code, ' @-@-@ ', $x/@english_name)";

		final Map<String, String> map = Maps.newHashMap();
		for (String s : serviceLocator.getService(ISLookUpService.class).quickSearchProfile(query)) {
			String[] arr = s.split("@-@-@");
			map.put(arr[0].trim(), arr[1].trim());
		}

		return map;
	}

	private Set<String> getParameterNamesForProtocol(final String protocol) {
		if (parametersMap.isEmpty()) {
			for (ProtocolDescriptor d : serviceLocator.getService(CollectorService.class).listProtocols()) {
				parametersMap.put(d.getName().toLowerCase(), d.getParams());
			}
		}
		final Set<String> res = Sets.newHashSet();
		if (parametersMap.containsKey(protocol.toLowerCase())) {
			res.add("baseUrl");
			res.addAll(parametersMap.get(protocol.toLowerCase()));
		}

		return res;
	}

	public String getBrowseFieldsJson() {
		return browseFieldsJson;
	}

	@Required
	public void setBrowseFieldsJson(final String browseFieldsJson) {
		this.browseFieldsJson = browseFieldsJson;
	}

}
