package eu.dnetlib.functionality.modular.ui.repositories;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.ui.ModelMap;

import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.gson.Gson;

import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.functionality.modular.ui.ModuleEntryPoint;
import eu.dnetlib.functionality.modular.ui.repositories.objects.VocabularyEntry;
import eu.dnetlib.miscutils.collections.Pair;

public class RepoApisEntryPointController extends ModuleEntryPoint {

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Resource
	private RepoUIUtils repoUIUtils;

	private String compatibilityLevelsVocabulary;

	public class RepoHIWorkflow implements Comparable<RepoHIWorkflow> {

		private String id;
		private String name;
		private Set<String> ifaceTypes;
		private Set<String> compliances;
		private List<Pair<String, String>> fields;

		/**
		 * Instantiates a new repo hi workflow.
		 *
		 * @param id
		 *            the id
		 * @param name
		 *            the name
		 * @param ifaceTypes
		 *            the iface types
		 * @param compliances
		 *            the compliances
		 * @param fields
		 *            the fields
		 */
		public RepoHIWorkflow(final String id, final String name, final Set<String> ifaceTypes, final Set<String> compliances,
				final List<Pair<String, String>> fields) {
			super();
			this.id = id;
			this.name = name;
			this.ifaceTypes = ifaceTypes;
			this.compliances = compliances;
			this.fields = fields;
		}

		public String getId() {
			return id;
		}

		public String getName() {
			return name;
		}

		public Set<String> getIfaceTypes() {
			return ifaceTypes;
		}

		public Set<String> getCompliances() {
			return compliances;
		}

		@Override
		public int compareTo(final RepoHIWorkflow o) {
			return getName().compareTo(o.getName());
		}

		/**
		 * @return the fields
		 */
		public List<Pair<String, String>> getFields() {
			return fields;
		}

		/**
		 * @param fields
		 *            the fields to set
		 */
		public void setFields(final List<Pair<String, String>> fields) {
			this.fields = fields;
		}

	}

	private RepoHIWorkflow parseQuery(final String input) {

		SAXReader reader = new SAXReader();
		try {
			Document doc = reader.read(new StringReader(input));
			Element rootNode = doc.getRootElement();
			Node node = doc.selectSingleNode("//id");
			String id = node.getText();
			node = doc.selectSingleNode("//name");
			String name = node != null ? node.getText() : "";
			node = doc.selectSingleNode("//types");
			String type = node != null ? node.getText() : "";
			final Set<String> ifcTypes = Sets.newHashSet(Splitter.on(",").omitEmptyStrings().trimResults().split(type));
			node = doc.selectSingleNode("//compliances");
			String compliance = node != null ? node.getText() : "";
			final Set<String> compliances = Sets.newHashSet(Splitter.on(",").omitEmptyStrings().trimResults().split(compliance));
			List<Node> nodes = doc.selectNodes(".//FIELD");
			List<Pair<String, String>> fields = new ArrayList<Pair<String, String>>();
			if (nodes != null) {
				for (Node currentNode : nodes) {
					String key = currentNode.valueOf("@name");
					String value = currentNode.getText();
					fields.add(new Pair<String, String>(key, value));
				}
			}
			return new RepoHIWorkflow(id, name, ifcTypes, compliances, fields);

		} catch (Exception e) {
			return null;
		}
	}

	private List<RepoHIWorkflow> listRepoHIWorkflows() throws ISLookUpException {

		final String xquery = "for $x in collection('/db/DRIVER/WorkflowDSResources/WorkflowDSResourceType') where $x//WORKFLOW_TYPE='REPO_HI' return <result> <id>{$x//RESOURCE_IDENTIFIER/@value/string()}</id> <name>{$x//WORKFLOW_NAME/text()}</name> <types>{$x//PARAM[@name='expectedInterfaceTypologyPrefixes']/text()}</types> <compliances>{$x//PARAM[@name='expectedCompliancePrefixes']/text()}</compliances> {$x//FIELD} </result>";

		// final String xquery = "for $x in collection('/db/DRIVER/WorkflowDSResources/WorkflowDSResourceType') "
		// + "where $x//WORKFLOW_TYPE='REPO_HI' "
		// +
		// "return concat($x//RESOURCE_IDENTIFIER/@value, ' @@@ ', $x//WORKFLOW_NAME, ' @@@ ', $x//PARAM[@name='expectedInterfaceTypologyPrefixes'], ' @@@ ', $x//PARAM[@name='expectedCompliancePrefixes'])";

		final List<RepoHIWorkflow> list = Lists.newArrayList();
		for (String s : serviceLocator.getService(ISLookUpService.class).quickSearchProfile(xquery)) {
			RepoHIWorkflow repoHiInfo = parseQuery(s);
			if (repoHiInfo != null) {
				list.add(repoHiInfo);
			}
		}
		Collections.sort(list);
		return list;
	}

	private List<VocabularyEntry> listCompatibilityLevels() throws ISLookUpException {
		final String xquery = "for $x in collection('/db/DRIVER/VocabularyDSResources/VocabularyDSResourceType')[.//VOCABULARY_NAME/@code = '"
				+ getCompatibilityLevelsVocabulary().trim() + "']//TERM return concat($x/@code, ' @@@ ', $x/@english_name)";

		final List<VocabularyEntry> list = Lists.newArrayList();
		for (String s : serviceLocator.getService(ISLookUpService.class).quickSearchProfile(xquery)) {
			final String[] arr = s.split("@@@");
			list.add(new VocabularyEntry(arr[0].trim(), arr[1].trim()));
		}
		Collections.sort(list);

		return list;
	}

	@Override
	protected void initialize(final ModelMap map, final HttpServletRequest request, final HttpServletResponse response) throws Exception {
		map.addAttribute("availableRepohiWfs", new Gson().toJson(listRepoHIWorkflows()));
		map.addAttribute("compatibilityLevels", new Gson().toJson(listCompatibilityLevels()));
		map.addAttribute("browseFields", repoUIUtils.getBrowseFieldsJson());
	}

	public String getCompatibilityLevelsVocabulary() {
		return compatibilityLevelsVocabulary;
	}

	@Required
	public void setCompatibilityLevelsVocabulary(final String compatibilityLevelsVocabulary) {
		this.compatibilityLevelsVocabulary = compatibilityLevelsVocabulary;
	}

}
