package eu.dnetlib.functionality.modular.ui.lightui.objects;

import java.io.StringReader;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;

import com.google.common.base.Function;
import com.google.common.collect.Lists;

import eu.dnetlib.functionality.index.client.response.BrowseEntry;
import eu.dnetlib.functionality.index.client.response.BrowseValueEntry;
import eu.dnetlib.functionality.modular.ui.lightui.LightUIUtils;
import eu.dnetlib.functionality.modular.ui.lightui.clients.ISLookupClient;
import eu.dnetlib.functionality.modular.ui.lightui.clients.IndexLightUIClient;

public class SearchFieldGrouper {

	private static final int maxBrowseValues = 100;

	@Resource
	private IndexLightUIClient indexClient;

	@Resource
	private ISLookupClient isLookupClient;

	@Resource
	private LightUIUtils lightUiUtils;

	private static final Log log = LogFactory.getLog(SearchFieldGrouper.class);

	public List<SearchField> getAllFields(final String lightuiId) {
		final List<SearchField> res = Lists.newArrayList();
		final SAXReader reader = new SAXReader();
		try {
			final Document doc = reader.read(new StringReader(isLookupClient.getLightUiProfile(lightuiId)));
			final Node searchNode = doc.selectSingleNode("//CONFIGURATION/SEARCH");
			final IndexConfiguration idx = getIndexConfiguration(searchNode);
			res.addAll(getPredefinedFields(searchNode));
			res.addAll(calculateBrowseFields(idx, searchNode));
		} catch (Exception e) {
			log.error("Error obtaining search fields", e);
		}
		return res;
	}

	private IndexConfiguration getIndexConfiguration(final Node searchNode) {
		final String id = searchNode.valueOf("./INDEX/@id");
		final String format = searchNode.valueOf("./INDEX/@format");
		final String layout = searchNode.valueOf("./INDEX/@layout");
		final String interpretation = searchNode.valueOf("./INDEX/@interpretation");
		final String backendId = searchNode.valueOf("./INDEX/@backendId");

		return new IndexConfiguration(id, format, layout, interpretation, backendId);
	}

	private List<SearchField> calculateBrowseFields(final IndexConfiguration idx, final Node node) {
		try {
			return Lists.transform(indexClient.browse("*=*", idx, lightUiUtils.getBrowseFields(node), maxBrowseValues),
					new Function<BrowseEntry, SearchField>() {

						@Override
						public SearchField apply(final BrowseEntry r) {
							final SearchField sf = new SearchField();
							sf.setField(r.getField());
							sf.setLabel(r.getLabel());
							sf.setOperator("exact");
							sf.setPredefinedValues(Lists.transform(r.getValues(), new Function<BrowseValueEntry, ValueWithLabel>() {

								@Override
								public ValueWithLabel apply(final BrowseValueEntry bv) {
									return new ValueWithLabel(bv.getValue() + " (" + bv.getSize() + ")", bv.getValue());
								}
							}));
							return sf;
						}
					});
		} catch (Exception e) {
			log.warn("Error executing default browse query", e);
			return Lists.newArrayList();
		}
	}

	private List<SearchField> getPredefinedFields(final Node node) {
		final List<SearchField> list = Lists.newArrayList();

		for (Object o : node.selectNodes(".//TEXT_FIELD")) {
			final String id = ((Element) o).valueOf("@id");
			final String label = ((Element) o).valueOf("@label");
			final String operator = ((Element) o).valueOf("@operator");
			list.add(new SearchField(id, label, operator));
		}

		for (Object o : node.selectNodes(".//SELECT_FIELD")) {
			final String id = ((Element) o).valueOf("@id");
			final String label = ((Element) o).valueOf("@label");
			final String operator = ((Element) o).valueOf("@operator");

			final List<ValueWithLabel> values = Lists.newArrayList();
			for (Object v : ((Element) o).selectNodes("./VALUE")) {
				final String v_id = ((Element) v).valueOf("@id");
				final String v_label = ((Element) v).valueOf("@label");
				values.add(new ValueWithLabel(v_label, v_id));
			}
			list.add(new SearchField(id, label, operator, values));
		}

		return list;
	}

}
