package eu.dnetlib.functionality.index.actors;

import java.io.IOException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.z3950.zing.cql.CQLParseException;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;

import eu.dnetlib.data.provision.index.rmi.IndexServiceException;
import eu.dnetlib.functionality.index.IndexCollection;
import eu.dnetlib.functionality.index.IndexServerDAO;
import eu.dnetlib.functionality.index.IndexServerDAOMap;
import eu.dnetlib.functionality.index.cql.CqlTranslator;
import eu.dnetlib.functionality.index.feed.DocumentFeeder;
import eu.dnetlib.functionality.index.feed.DocumentMapperFactory;
import eu.dnetlib.functionality.index.feed.FeedMode;
import eu.dnetlib.functionality.index.feed.FeedResult;
import eu.dnetlib.functionality.index.model.document.IndexDocument;
import eu.dnetlib.functionality.index.utils.IndexDateUtility;
import eu.dnetlib.functionality.index.utils.IndexFieldUtility;
import eu.dnetlib.functionality.index.utils.MetadataReference;
import eu.dnetlib.functionality.index.utils.ServiceTools;
import eu.dnetlib.miscutils.datetime.DateUtils;

/**
 * The Class IndexFeedActorImpl.
 */
public class IndexFeedActorImpl implements IndexFeedActor {

	/** The Constant log. */
	private static final Log log = LogFactory.getLog(IndexFeedActorImpl.class); // NOPMD by marko on 11/24/08 5:02 PM

	/** The index server dao map. */
	private final transient IndexServerDAOMap indexServerDAOMap;

	/** The service tools. */
	private final transient ServiceTools serviceTools;

	/** CqlTranslator. */
	private CqlTranslator translator;

	/**
	 * Thread pool used for the feeding process.
	 */
	private final transient ExecutorService threadPool = Executors.newCachedThreadPool();

	/**
	 * Instantiates a new index feed actor impl.
	 * 
	 * @param indexServerDAOMap
	 *            the index server dao map
	 * @param serviceTools
	 *            the service tools
	 */
	public IndexFeedActorImpl(final IndexServerDAOMap indexServerDAOMap, final ServiceTools serviceTools, final CqlTranslator translator) {
		super();
		this.indexServerDAOMap = indexServerDAOMap;
		this.serviceTools = serviceTools;
		this.translator = translator;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.functionality.index.actors.IndexFeedActor#feedIndex(java.lang.String, eu.dnetlib.functionality.index.feed.FeedMode,
	 *      java.lang.Iterable, eu.dnetlib.functionality.index.actors.ResultsetKeepAliveCallback,
	 *      eu.dnetlib.functionality.index.actors.BlackboardActorCallback)
	 */
	@Override
	public void feedIndex(final String dsId,
			final FeedMode feedMode,
			final Iterable<String> docIterator,
			final ResultsetKeepAliveCallback startCallback,
			final BlackboardActorCallback endCallback,
			final String backendId) {
		IndexCollection indexCollection = null;
		try {
			startCallback.unschedule();
			final MetadataReference mdref = serviceTools.getMetadataRef(dsId);

			final IndexServerDAO serverDAO = indexServerDAOMap.getIndexServerDAO(backendId);
			final DocumentMapperFactory docMapperFactory = serverDAO.getDocumentMapperFactory();
			final String version = DateUtils.now_ISO8601();
			final Function<String, IndexDocument> docMapper = docMapperFactory.getRecordMapper(serverDAO.getSchema(mdref), mdref, dsId, version);
			Iterable<IndexDocument> result = Iterables.transform(docIterator, docMapper);
			indexCollection = serverDAO.getIndexCollection(mdref);
			final FeedResult res = threadPool.submit(new DocumentFeeder(indexCollection, result)).get();

			cleanMarkedDocuments(indexCollection, dsId);
			if (feedMode.equals(FeedMode.REFRESH)) {
				deleteByVersion(indexCollection, dsId, version);
			}

			indexCollection.commit();
			indexCollection.shutdown();

			log.info("FeedResult: " + res.setTimeElapsed(System.currentTimeMillis()));
			endCallback.setJobDone();
		} catch (final Exception e) {
			endCallback.setJobFailed(e);
			log.error("feed index job failed", e);
		} finally {
			if (indexCollection != null) {
				indexCollection.shutdown();
			}
		}

	}

	/**
	 * method deletes all the documents of a specified dsId whose {@link IndexMap}.DS_VERSION field is older than the specified
	 * mdFormatVersion.
	 * 
	 * @param indexCollection
	 *            the server dao
	 * @param dsId
	 *            the ds id
	 * @param version
	 *            the version
	 * @throws IndexServiceException
	 *             the index service exception
	 */
	private void deleteByVersion(final IndexCollection indexCollection, final String dsId, final String version) throws IndexServiceException {
		final String cqlQuery = IndexFieldUtility.DS_VERSION + " < \"" + IndexDateUtility.getParsedDateField(version) + "\"";
		try {
			String luceneQuerty = translator.getTranslatedQuery(cqlQuery).asLucene();
			indexCollection.deleteByQuery(luceneQuerty, dsId);
		} catch (CQLParseException e) {
			throw new IndexServiceException("Cannot parse CQL query into lucene query: " + cqlQuery, e);
		} catch (IOException e) {
			throw new IndexServiceException("Cannot parse CQL query into lucene query: " + cqlQuery, e);
		}

	}

	/**
	 * method delete documents where IndexMap.DELETE_DOCUMENT field is true
	 * 
	 * @param indexCollection
	 *            the server dao
	 * @param dsId
	 *            the ds id
	 * @return the time elapsed to complete the operation.
	 * @throws IndexServiceException
	 *             the index service exception
	 */
	public void cleanMarkedDocuments(final IndexCollection indexCollection, final String dsId) throws IndexServiceException {

		final String cqlQuery = IndexFieldUtility.DELETE_DOCUMENT + " all true ";
		try {
			String luceneQuery = translator.getTranslatedQuery(cqlQuery).asLucene();
			indexCollection.deleteByQuery(luceneQuery, dsId);
		} catch (CQLParseException e) {
			throw new IndexServiceException("Cannot parse CQL query into lucene query: " + cqlQuery, e);
		} catch (IOException e) {
			throw new IndexServiceException("Cannot parse CQL query into lucene query: " + cqlQuery, e);
		}
	}

}
