package eu.dnetlib.functionality.modular.ui.db;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.servlet.ServletOutputStream;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.IOUtils;
import org.springframework.jdbc.support.rowset.SqlRowSet;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import eu.dnetlib.enabling.database.DatabaseServiceCore;
import eu.dnetlib.enabling.database.objects.DnetDatabase;
import eu.dnetlib.enabling.database.resultset.IterableRowSet;
import eu.dnetlib.enabling.resultset.client.utils.EPRUtils;

@Controller
public class DbManagerInternalController {
	public static final String CTYPE_CSS = "text/css";
	public static final String CTYPE_JS = "text/javascript";
	public static final String CTYPE_XML = "text/xml";
	public static final String CTYPE_TEXT = "text/plain";
	public static final String CTYPE_HTML = "text/html";
	
	@Resource
	private DatabaseServiceCore core;

	@RequestMapping("/ui/listDBs.do")
	public @ResponseBody List<DnetDatabase> listDatabases() throws Exception {
		return core.listDatabases();
	}

	@RequestMapping("/ui/listTables.do")
	public @ResponseBody List<Map<String, Object>> listTables(@RequestParam(value = "db", required = true) String db) throws Exception {
		
		final List<Map<String, Object>> tables = new ArrayList<Map<String, Object>>();
		final Map<String, String> logSize = new HashMap<String, String>();
		
		final String sql = IOUtils.toString(getClass().getResourceAsStream("tables_info.sql"));
		
		final SqlRowSet rows = core.getDbUtils().executeSql(db, sql, SqlRowSet.class);
		while (rows.next()) {
			final String tname = rows.getString("name");
			if (tname.endsWith("_log")) {
				logSize.put(tname.substring(0, tname.length() - 4), rows.getString("total"));
			} else {
				final Map<String, Object> t = new HashMap<String, Object>();
				t.put("name", rows.getString("name"));
				t.put("view", rows.getString("kind").equalsIgnoreCase("v"));
				t.put("data", rows.getString("data"));
				t.put("indices", rows.getString("indices"));
				t.put("total", rows.getString("total"));
				t.put("managed", core.getDbUtils().isManagedTable(db, tname));
				t.put("logged", core.getDbUtils().isLoggedTable(db, tname));
				t.put("dnetIdentifier", core.getDbUtils().getDefaultDnetIdentifier(db, tname));
				tables.add(t);
			}
		}
		for (Map<String, Object> t : tables) {
			if (logSize.containsKey(t.get("name"))) {
				t.put("logdata", logSize.get(t.get("name")));
			} else {
				t.put("logdata", "-");
			}
		}
		
		return tables;
	}

	@RequestMapping("/ui/manageDB.do")
	public @ResponseBody boolean manageDB(@RequestParam(value = "db", required = true) String db,
			@RequestParam(value = "manage", required = true) boolean b) throws Exception {
		core.changeDatabaseStatus(db, b);
		return true;
	}

	@RequestMapping("/ui/importEPR.do")
	public @ResponseBody boolean importEPR(@RequestParam(value = "db", required = true) String db,
			@RequestParam(value = "epr", required = true) String epr) throws Exception {

		core.importFromResultset(db, (new EPRUtils()).getEpr(epr));

		return true;
	}
	
	
	@RequestMapping("/ui/query.do")
	public @ResponseBody List<String> query(@RequestParam(value = "query", required = true) final String query,
			@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "limit", required = true) int limit) throws Exception {
		
		return Lists.newArrayList(Iterables.limit(new IterableRowSet(db, query, null, core.getDbUtils()), limit));
	
	}
	
	@RequestMapping("/ui/changeTableManagement.do")
	public @ResponseBody boolean changeTableManagement(@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = true) final String t) throws Exception {
		
		if (core.getDbUtils().isManagedTable(db, t)) {
			core.getDbUtils().removeManagementOfTable(db, t);
		} else {
			core.getDbUtils().prepareManagementOfTable(db, t);			
		}
		
		return true;
	} 

	
	@RequestMapping("/ui/changeIdentifiers.do")
	public @ResponseBody boolean changeId(HttpServletResponse res, 
			@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = false) final String t) throws Exception {
		
		if (t == null || t.isEmpty()) {
			core.getDbUtils().reassignDefaultDnetIdentifiers(db);
		} else {
			core.getDbUtils().reassignDefaultDnetIdentifiers(db, t);
		}
		
		return true;
	} 
	
	@RequestMapping("/ui/changeTableLog.do")
	public @ResponseBody boolean changeTableLog(@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = true) final String t) throws Exception {

		if (core.getDbUtils().isManagedTable(db, t)) {
			if (core.getDbUtils().isLoggedTable(db, t)) {
				core.getDbUtils().removeLogTable(db, t);
			} else {
				core.getDbUtils().addLogTable(db, t);			
			}
		}

		return true;
	} 

	@RequestMapping("/ui/describeTable.do")
	public @ResponseBody List<Map<?, ?>> describeTable(@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = true) final String t) throws Exception {
		return core.getDbUtils().describeTable(db, t);
	} 

	@RequestMapping("/ui/dumpTable.do")
	public void dumpTable(final ServletResponse response, @RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = true) final String t) throws Exception {
		sendXML(response, core.getDbUtils().dumpTableAsXML(db,t));
	}

	@RequestMapping("/ui/dumpTableEPR.do")
	public void dumpTableEPR(final ServletResponse response, 
			@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = true) final String t) throws Exception {
		
		sendXML(response, core.generateResultSet(db, t, null).toString());
	}
	
	
	public void sendXML(final ServletResponse response, final String xml) throws IOException {
		response.setContentType("text/xml");
		
		final ServletOutputStream out = response.getOutputStream();
		IOUtils.copy(new StringReader(xml), out);
		
		out.flush();
		out.close();
	}
}
